﻿using System.Linq.Expressions;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo;

public interface IDemo
{
    Task Run(DevelopmentWorkspace workspace, TestConfiguration configuration);
}

public abstract class DemoBase<TDelegate, TEntry, TResult> : IDemo where TDelegate : Delegate
{
    public abstract string DemoName { get; }
    
    public abstract Expression<Action<TDelegate,TEntry>> ArgumentsMapping { get; }
    
    public abstract Func<TEntry,TResult> ResultMapping { get; }
    
    public abstract Func<TResult,TResult,bool>? ResultVerifier { get; }
    
    protected abstract TEntry[] PrepareData();
    
    protected abstract void ConfigureTrainer(Trainer<TDelegate, TEntry, TResult> trainer);

    protected virtual void ConfigureRag(MockFunction mock) {}
    
    public async Task Run(DevelopmentWorkspace workspace, TestConfiguration configuration)
    {
        var mock = workspace.NewObject<MockDelegate<TDelegate>>();
        var logger = mock.AttachComponent<LoggerComponent>();
        logger.AuthorName = DemoName;

        var data = PrepareData();
        
        var trainer = new Trainer<TDelegate, TEntry, TResult>(mock, data,
            ArgumentsMapping, ResultMapping, ResultVerifier)
        {
            Logger = mock.GetLogger(),
        };
        ConfigureTrainer(trainer);
        ConfigureRag(mock);

        mock.Script.OnCodeChanged += code =>
        {
            trainer.Logger?.LogDetails(LogLevel.Information, "Script Changed.",
                details => { details.ScriptCode = code ?? "[Empty]"; });
        };

        var trainingCount = configuration.TrainingCount;
        var evaluationCount = data.Length - trainingCount;
        mock.EnableStructuredOutput = configuration.EnableStructuredOutput;
        mock.EnableSubstitutionScript = configuration.EnableSubstitutionScript;
        trainer.CompressionThreshold = configuration.CompressionThreshold;
        trainer.ReplacementThreshold = configuration.ReplacementThreshold;

        Console.Title = $"{DemoName} - {configuration.ModelName} - {configuration.TrainingCount}" +
                        $"{(configuration.EnableSubstitutionScript? " - Script" : "")}";
        AnsiConsole.MarkupLine($"[green]{DemoName}:[/] [blue]{configuration.ModelName}[/] - " +
                               $"{configuration.TrainingCount}" +
                               $"{(configuration.EnableSubstitutionScript? " - [green]Script[/]" : "")}");
        AnsiConsole.MarkupLine($"[bold]Session GUID:[/] [blue]{mock.Identifier}[/]");

        trainer.Logger?.LogDetails(LogLevel.Information, "Experiment Description",
            details =>
            {
                details.ModelName = configuration.ModelName;
                details.TrainingCount = trainingCount;
                details.TestCount = evaluationCount;
                details.WithScriptAcceleration = mock.EnableSubstitutionScript;
                details.CompressionThreshold = trainer.CompressionThreshold!;
                details.ReplacementThreshold = trainer.ReplacementThreshold!;
            });

        await trainer.Train(trainingCount);

        if (mock.EnableSubstitutionScript)
        {
            while (mock.Script.Code == null)
            {
                if (await mock.Script.Build())
                    break;
            }
        }
        
        await trainer.Evaluate(evaluationCount);
        
        AnsiConsole.MarkupLine($"[green]{DemoName}:[/] [blue]{configuration.ModelName}[/] - " +
                               $"{configuration.TrainingCount}" +
                               $"{(configuration.EnableSubstitutionScript? " - " +
                                                                           "[green]Script[/]" : "")}");
        AnsiConsole.MarkupLine($"[bold]Session GUID:[/] [blue]{mock.Identifier}[/]");
        
        Console.Beep();
    }
}