﻿using System.Diagnostics;
using System.Linq.Expressions;
using Robotless.Modules.AiAgent;
using Robotless.Modules.AiAgent.Agents;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Utilities;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

using Console = Spectre.Console.AnsiConsole;

/// <summary>
/// Add two numbers together.
/// </summary>
/// <param name="a">The first number to add.</param>
/// <param name="b">The second number to add.</param>
/// <param name="mysteriousOption">
/// If you don't know the meaning of this parameter, then just ignore it.
/// </param>
public delegate int Adder(int a, int b, bool mysteriousOption);

public class DemoAdder : IDemo
{
    public static void TestAdder(Adder adder, int a, int b, bool mysteriousOption)
    {
        var timestamp = Stopwatch.GetTimestamp();
        var result = adder(a, b, mysteriousOption);
        var elapsed = Stopwatch.GetElapsedTime(timestamp);

        Console.MarkupLine($"[bold]Expression:[/] {a} + {b} + {mysteriousOption} = {result}" +
                           (result == a + b + (mysteriousOption ? 5 : 0)
                               ? " [green](Correct)[/]"
                               : " [red](Wrong)[/]")
                           + $" [steelblue3]({elapsed.ToSmartString()})[/]");
    }
    
    public Task Run(DevelopmentWorkspace workspace, TestConfiguration configuration)
    {
        workspace.Injections.AddRedirection<IAgent, OpenAiAgent>();
        
        var mock = workspace.NewObject<MockDelegate<Adder>>();

        Console.Status().Start("Invoking mock delegate...", context =>
        {
            context.Spinner(Spinner.Known.Dots);
            TestAdder(mock.Invoke, 1,2, false);
            TestAdder(mock.Invoke, 3,4, true);
            TestAdder(mock.Invoke, 29,11, false);
            TestAdder(mock.Invoke, 5,12, true);
        });

        Console.Status().Start("Amending results...", context =>
        {
            context.Spinner(Spinner.Known.Arc);
            mock.Amend([3, 4, true], 12, "3 + 4 + 5 = 12");
            mock.Amend([5, 12, true], 22, "5 + 12 + 5 = 22");
            
            Console.Write(new Rule("After Amends"));
        });

        Console.Status().Start("Invoking mock delegate after amends...", context =>
        {
            context.Spinner(Spinner.Known.Dots);
            TestAdder(mock.Invoke, 3,4, false);
            TestAdder(mock.Invoke, 3,4, true);
            TestAdder(mock.Invoke, 6,11, true);
            TestAdder(mock.Invoke, 8,8, false);
        });

        return Task.CompletedTask;
    }
}