﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class CarEntry
{
    // [Name("id")] public int Id { get; set; }

    [Name("brand")] public string Brand { get; set; }

    [Name("model")] public string Model { get; set; }

    [Name("model_year")] public int ModelYear { get; set; }

    [Name("milage")] public string Milage { get; set; }

    [Name("fuel_type")] public string FuelType { get; set; }

    [Name("engine")] public string Engine { get; set; }

    [Name("transmission")] public string Transmission { get; set; }

    [Name("ext_col")] public string ExteriorColor { get; set; }

    [Name("int_col")] public string InteriorColor { get; set; }

    [Name("accident")] public string Accident { get; set; }

    [Name("clean_title")] public string? CleanTitle { get; set; }

    [Name("price")] public string Price { get; set; }
}

/// <summary>
/// Estimate the price of a car based on its conditions.
/// </summary>
/// <param name="brand">The brand or the company name of this car.</param>
/// <param name="model">The model name of this car.</param>
/// <param name="modelYear">
/// The manufacturing year of the vehicles,
/// crucial for assessing depreciation and technology advancements.
/// </param>
/// <param name="milage">
/// The mileage of each vehicle, a key indicator of wear and tear and potential maintenance requirements.
/// </param>
/// <param name="fuelType">
/// The type of fuel the vehicles run on, whether it's gasoline, diesel, electric, or hybrid.
/// </param>
/// <param name="engine">Understand the engine specifications, shedding light on performance and efficiency.</param>
/// <param name="transmission">The transmission type, whether automatic, manual, or another variant.</param>
/// <param name="exteriorColor">
/// The exterior color of this car, indicating the aesthetic aspects of the vehicles.
/// </param>
/// <param name="interiorColor">
/// The interior color of this car, indicating the aesthetic aspects of the vehicles.
/// </param>
/// <param name="accident">
/// Whether a vehicle has a prior history of accidents or damage, crucial for informed decision-making.
/// </param>
/// <param name="cleanTitle">
/// The availability of a clean title, which can impact the vehicle's resale value and legal status.
/// It will be null if whether this car has a clean title is unknown.
/// </param>
public delegate int EstimateCarPrice(string brand, string model, int modelYear,
    string milage, string fuelType, string engine, string transmission,
    string exteriorColor, string interiorColor, string accident, bool? cleanTitle);

public class DemoCarPrices : DemoBase<EstimateCarPrice, CarEntry, int>
{
    public override string DemoName { get; } = "Car Prices Regression";

    public override Expression<Action<EstimateCarPrice, CarEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(entry.Brand, entry.Model, entry.ModelYear,
            entry.Milage, entry.FuelType, entry.Engine, entry.Transmission,
            entry.ExteriorColor, entry.InteriorColor, entry.Accident,
            entry.CleanTitle != null ? entry.CleanTitle == "Yes" : null);

    public override Func<CarEntry, int> ResultMapping { get; } =
        entry => int.Parse(entry.Price.Replace("$", "").Replace(",", ""));

    public static double TolerationRatio { get; set; } = 0.0;

    public override Func<int, int, bool>? ResultVerifier { get; } =
        (expected, actual) => Math.Abs(expected - actual) < expected * TolerationRatio;

    protected override CarEntry[] PrepareData()
    {
        return DataLoader.FromCsv<CarEntry>("../CarPrices.csv").Take(500).Shuffle();
    }

    protected override void ConfigureTrainer(Trainer<EstimateCarPrice, CarEntry, int> trainer)
    {
        trainer.Metrics
            .UseLambdaMetrics("ReflectionThreshold",
                (expected, _) => expected * TolerationRatio)
            .UseStandardDeviation()
            .UseRootMeanSquareError()
            .UseErrorRatio();
    }
}