﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class HorseEntry
{
    [Name("surgery")] public string Surgery { get; set; }

    [Name("age")] public string Age { get; set; }

    [Name("hospital_number")] public string HospitalNumber { get; set; }

    [Name("rectal_temp")] public string RectalTemperature { get; set; }

    [Name("pulse")] public string Pulse { get; set; }

    [Name("respiratory_rate")] public string RespiratoryRate { get; set; }

    [Name("temp_of_extremities")] public string TemperatureOfExtremities { get; set; }

    [Name("peripheral_pulse")] public string PeripheralPulse { get; set; }

    [Name("mucous_membrane")] public string MucousMembranes { get; set; }

    [Name("capillary_refill_time")] public string CapillaryRefillTime { get; set; }

    [Name("pain")] public string Pain { get; set; }

    [Name("peristalsis")] public string Peristalsis { get; set; }

    [Name("abdominal_distention")] public string AbdominalDistension { get; set; }

    [Name("nasogastric_tube")] public string NasogastricTube { get; set; }

    [Name("nasogastric_reflux")] public string NasogastricReflux { get; set; }

    [Name("nasogastric_reflux_ph")] public string NasogastricRefluxPh { get; set; }

    [Name("rectal_exam_feces")] public string RectalExamination { get; set; }

    [Name("abdomen")] public string Abdomen { get; set; }

    [Name("packed_cell_volume")] public string PackedCellVolume { get; set; }

    [Name("total_protein")] public string TotalProtein { get; set; }

    [Name("abdomo_appearance")] public string AbdominocentesisAppearance { get; set; }

    [Name("abdomo_protein")] public string AbdominocentesisProtein { get; set; }

    [Name("outcome")] public string Outcome { get; set; }

    [Name("surgical_lesion")] public string SurgicalLesion { get; set; }

    [Name("lesion_1")] public string Lesion1 { get; set; }

    [Name("lesion_2")] public string Lesion2 { get; set; }

    [Name("lesion_3")] public string Lesion3 { get; set; }

    [Name("cp_data")] public string CpData { get; set; }
}

public enum HorseOutcome
{
    Lived,
    Died,
    Euthanized
}

/// <summary>
/// Predict the health outcome of the horse.
/// </summary>
/// <returns>
/// The predicted outcome of the horse.
/// </returns>
public delegate HorseOutcome PredictHorseOutcome(string surgery, string age, string hospitalNumber,
    string rectalTemperature,
    string pulse, string respiratoryRate, string temperatureOfExtremities, string peripheralPulse,
    string mucousMembranes,
    string capillaryRefillTime, string pain, string peristalsis, string abdominalDistension, string nasogastricTube,
    string nasogastricReflux, string nasogastricRefluxPh, string rectalExamination, string abdomen,
    string packedCellVolume,
    string totalProtein, string abdominocentesisAppearance, string abdominocentesisProtein,
    string surgicalLesion, string lesion1, string lesion2, string lesion3, string cpData);

public class DemoHorses : DemoBase<PredictHorseOutcome, HorseEntry, HorseOutcome>
{
    public override string DemoName { get; } = "Horse Health Outcome Classifier";

    public override Expression<Action<PredictHorseOutcome, HorseEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(entry.Surgery, entry.Age, entry.HospitalNumber, entry.RectalTemperature,
            entry.Pulse, entry.RespiratoryRate, entry.TemperatureOfExtremities, entry.PeripheralPulse,
            entry.MucousMembranes, entry.CapillaryRefillTime, entry.Pain, entry.Peristalsis,
            entry.AbdominalDistension, entry.NasogastricTube, entry.NasogastricReflux,
            entry.NasogastricRefluxPh, entry.RectalExamination, entry.Abdomen, entry.PackedCellVolume,
            entry.TotalProtein, entry.AbdominocentesisAppearance, entry.AbdominocentesisProtein,
            entry.SurgicalLesion, entry.Lesion1, entry.Lesion2, entry.Lesion3, entry.CpData);


    public override Func<HorseEntry, HorseOutcome> ResultMapping { get; } =
        entry => Enum.Parse<HorseOutcome>(entry.Outcome, true);

    public override Func<HorseOutcome, HorseOutcome, bool>? ResultVerifier { get; }
        = (expected, actual) => expected == actual;

    protected override HorseEntry[] PrepareData()
    {
        return DataLoader.FromCsv<HorseEntry>("../HorseHealth.csv")
            .Take(1000).Shuffle();
    }

    protected override void ConfigureTrainer(Trainer<PredictHorseOutcome, HorseEntry, HorseOutcome> trainer)
    {
        trainer.Metrics
            .UseAccuracy()
            .UseCorrectness();
    }
}