﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class InsuranceEntry
{
    [Name("Gender")]
    public string Gender { get; set; }
    
    [Name("Age")]
    public int Age { get; set; }
    
    [Name("Driving_License")]
    public int DrivingLicense { get; set; }
    
    [Name("Region_Code")]
    public double RegionCode { get; set; }
    
    [Name("Previously_Insured")]
    public int PreviouslyInsured { get; set; }
    
    [Name("Vehicle_Age")]
    public string VehicleAge { get; set; }
    
    [Name("Vehicle_Damage")]
    public string VehicleDamage { get; set; }
    
    [Name("Annual_Premium")]
    public double AnnualPremium { get; set; }
    
    [Name("Policy_Sales_Channel")]
    public double PolicySalesChannel { get; set; }
    
    [Name("Vintage")]
    public double Vintage { get; set; }
    
    [Name("Response")]
    public int Response { get; set; }
}

public enum InsuranceRespond
{
    No = 0,
    Yes = 1
}

/// <summary>
/// Predict the obesity type of the person.
/// </summary>
/// <returns>
/// The predicted obesity type of the person.
/// </returns>
public delegate InsuranceRespond PredictInsuranceRespond(string gender, int age, int drivingLicense, double regionCode,
    int previouslyInsured, string vehicleAge, string vehicleDamage, double annualPremium, double policySalesChannel,
    double vintage);

public class DemoInsurance : DemoBase<PredictInsuranceRespond, InsuranceEntry, InsuranceRespond>
{
    public override string DemoName { get; } = "Insurance Response Classifier";

    public override Expression<Action<PredictInsuranceRespond, InsuranceEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(entry.Gender, entry.Age, entry.DrivingLicense, entry.RegionCode,
            entry.PreviouslyInsured, entry.VehicleAge, entry.VehicleDamage, entry.AnnualPremium,
            entry.PolicySalesChannel, entry.Vintage);

    public override Func<InsuranceEntry, InsuranceRespond> ResultMapping { get; }
        = entry => entry.Response == 1 ? InsuranceRespond.Yes : InsuranceRespond.No;

    public override Func<InsuranceRespond, InsuranceRespond, bool>? ResultVerifier { get; }
        = (expected, actual) => expected == actual;
    
    protected override InsuranceEntry[] PrepareData()
    {
        return DataLoader.FromCsv<InsuranceEntry>("../InsuranceSelling.csv")
            .Take(1000).Shuffle();
    }

    protected override void ConfigureTrainer(Trainer<PredictInsuranceRespond, InsuranceEntry, InsuranceRespond> trainer)
    {
        trainer.Metrics
            .UseAccuracy()
            .UseCorrectness();
    }
}