﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.AiAgent;
using Robotless.Modules.AiAgent.Agents;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class IrisEntry
{
    [Name("sepal_length")] public double SepalLength { get; set; }

    [Name("sepal_width")] public double SepalWidth { get; set; }

    [Name("petal_length")] public double PetalLength { get; set; }

    [Name("petal_width")] public double PetalWidth { get; set; }

    [Name("species")] public string Species { get; set; }
}

public enum IrisSpecies
{
    Setosa,
    Versicolor,
    Virginica,
    Unknown
}

/// <summary>
/// Get the species of the iris, according to the sepal and petal sizes.
/// </summary>
/// <param name="sepalLength">Length of sepal. Can be null.</param>
/// <param name="sepalWidth">Width of sepal. Can be null.</param>
/// <param name="petalLength">Length of petal. Can be null.</param>
/// <param name="petalWidth">Width of petal. Can be null.</param>
/// <returns>
/// Species of the flower.
/// </returns>
public delegate IrisSpecies GetIrisSpecies(double? sepalLength, double? sepalWidth, double? petalLength,
    double? petalWidth);

public class DemoIris : DemoBase<GetIrisSpecies, IrisEntry, IrisSpecies>
{
    public override string DemoName { get; } = "Iris Species Classifier";

    public override Expression<Action<GetIrisSpecies, IrisEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(
            entry.SepalLength, entry.SepalWidth, entry.PetalLength, entry.PetalWidth);

    public override Func<IrisEntry, IrisSpecies> ResultMapping { get; }
        = entry => Enum.Parse<IrisSpecies>(entry.Species, true);
    
    public override Func<IrisSpecies, IrisSpecies, bool>? ResultVerifier { get; }
        = (expected, actual) => expected == actual;
        
    protected override IrisEntry[] PrepareData()
    {
        return DataLoader.FromCsvShuffled<IrisEntry>("../Iris.csv");
    }

    protected override void ConfigureTrainer(Trainer<GetIrisSpecies, IrisEntry, IrisSpecies> trainer)
    {
        trainer.Metrics
            .UseAccuracy()
            .UseCorrectness();
    }
}