﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class MohsEntry
{
    public string Formula { get; set; }
    
    [Name("Crystal structure")]
    public string CrystalSystem { get; set; }
    
    [Name("Hardness (Mohs)")]
    public double Hardness { get; set; }
    
    public double allelectrons_Total { get; set; }
    
    public double density_Total { get; set; }
    
    public double allelectrons_Average { get; set; }

    public double val_e_Average { get; set; }
    
    public double atomicweight_Average { get; set; }
    
    public double ionenergy_Average { get; set; }

    public double el_neg_chi_Average { get; set; }
    
    public double R_vdw_element_Average { get; set; }
    
    public double R_cov_element_Average { get; set; }
    
    public double zaratio_Average { get; set; }
    
    public double density_Average { get; set; }
}

/// <summary>
/// Estimate the mohs hardness of the crystal.
/// </summary>
public delegate double EstimateHardness(string formula, string crystalSystem,
    double allelectronsTotal, double densityTotal, double allelectronsAverage,
    double valEAverage, double atomicweightAverage, double ionenergyAverage,
    double elNegChiAverage, double rVdwElementAverage, double rCovElementAverage,
    double zaratioAverage, double densityAverage);

public class DemoMohs : DemoBase<EstimateHardness, MohsEntry, double>
{
    public override string DemoName { get; } = "Mohs Hardness Regression";

    public override Expression<Action<EstimateHardness, MohsEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(entry.Formula, entry.CrystalSystem, entry.allelectrons_Total,
            entry.density_Total,
            entry.allelectrons_Average, entry.val_e_Average, entry.atomicweight_Average, entry.ionenergy_Average,
            entry.el_neg_chi_Average, entry.R_vdw_element_Average, entry.R_cov_element_Average,
            entry.zaratio_Average, entry.density_Average);

    public override Func<MohsEntry, double> ResultMapping { get; } = entry => entry.Hardness;

    public static double TolerantDeviationRatio { get; set; } = 0.0;
    
    public override Func<double, double, bool>? ResultVerifier { get; } = (expected, actual) =>
        Math.Abs(expected - actual) < expected * TolerantDeviationRatio;
    protected override MohsEntry[] PrepareData()
    {
        return DataLoader.FromCsv<MohsEntry>(
            "../MohsHardness.csv").Shuffle();
    }

    protected override void ConfigureTrainer(Trainer<EstimateHardness, MohsEntry, double> trainer)
    {
        trainer.Metrics
            .UseLambdaMetrics("ReflectionThreshold",
                (expected, _) => expected * TolerantDeviationRatio)
            .UseStandardDeviation()
            .UseMedianAbsoluteError()
            .UseRootMeanSquareError()
            .UseErrorRatio();
    }
}