﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class MushroomEntry
{
    [Name("class")] 
    public string Type { get; set; }

    [Name("cap-shape")] 
    public string CapShape { get; set; }

    [Name("cap-surface")] 
    public string CapSurface { get; set; }

    [Name("cap-color")] 
    public string CapColor { get; set; }

    [Name("bruises")] 
    public string Bruises { get; set; }
    
    [Name("odor")]
    public string Odor { get; set; }
    
    [Name("gill-attachment")]
    public string GillAttachment { get; set; }
    
    [Name("gill-spacing")]
    public string GillSpacing { get; set; }
    
    [Name("gill-size")]
    public string GillSize { get; set; }
    
    [Name("gill-color")]
    public string GillColor { get; set; }
    
    [Name("stalk-shape")]
    public string StalkShape { get; set; }
    
    [Name("stalk-root")]
    public string StalkRoot { get; set; }
    
    [Name("stalk-surface-above-ring")]
    public string StalkSurfaceAboveRing { get; set; }
    
    [Name("stalk-surface-below-ring")]
    public string StalkSurfaceBelowRing { get; set; }
    
    [Name("veil-type")]
    public string VeilType { get; set; }
    
    [Name("veil-color")]
    public string VeilColor { get; set; }
    
    [Name("ring-number")]
    public string RingNumber { get; set; }
    
    [Name("ring-type")]
    public string RingType { get; set; }
    
    [Name("spore-print-color")]
    public string SporePrintColor { get; set; }
    
    [Name("population")]
    public string Population { get; set; }
    
    [Name("habitat")]
    public string Habitat { get; set; }
}

public enum MushroomType
{
    Poisonous,
    Edible
}

/// <summary>
/// Check if the mushroom is poisonous or edible.
/// </summary>
/// <returns>
/// Types of the mushroom, marking as poisonous or edible.
/// </returns>
public delegate MushroomType GetMushroomType(string capShape, string capSurface, string capColor,
    string bruises, string odor, string gillAttachment, string gillSpacing, string gillSize, string gillColor,
    string stalkShape, string stalkRoot, string stalkSurfaceAboveRing, string stalkSurfaceBelowRing, string veilType,
    string veilColor, string ringNumber, string ringType, string sporePrintColor, string population, string habitat);

public class DemoMushroom : DemoBase<GetMushroomType, MushroomEntry, MushroomType>
{
    public override string DemoName { get; } = "Mushroom Poisonous Classifier";

    public override Expression<Action<GetMushroomType, MushroomEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(entry.CapShape, entry.CapSurface, entry.CapColor,
            entry.Bruises, entry.Odor,
            entry.GillAttachment, entry.GillSpacing, entry.GillSize, entry.GillColor,
            entry.StalkShape, entry.StalkRoot, entry.StalkSurfaceAboveRing, entry.StalkSurfaceBelowRing,
            entry.VeilType, entry.VeilColor,
            entry.RingNumber, entry.RingType,
            entry.SporePrintColor,
            entry.Population,
            entry.Habitat);

    public override Func<MushroomEntry, MushroomType> ResultMapping { get; }
        = entry => entry.Type == "p" ? MushroomType.Poisonous : MushroomType.Edible;

    public override Func<MushroomType, MushroomType, bool>? ResultVerifier { get; }
        = (expected, actual) => expected == actual;
    protected override MushroomEntry[] PrepareData()
    {
        return DataLoader.FromCsv<MushroomEntry>("../Mushrooms.csv")
            .Take(1000).Shuffle();
    }

    protected override void ConfigureTrainer(Trainer<GetMushroomType, MushroomEntry, MushroomType> trainer)
    {
        trainer.Metrics
            .UseAccuracy()
            .UseCorrectness();
    }
}