﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using CsvHelper.Configuration.Attributes;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class TitanicEntry
{
    public int PassengerId { get; set; }

    public bool Survived { get; set; }

    [Name("Pclass")] public int PassengerClass { get; set; }

    public string Name { get; set; }

    [Name("Sex")] public string Gender { get; set; }

    public double? Age { get; set; }

    [Name("SibSp")] public int SiblingOnBoard { get; set; }

    [Name("Parch")] public int ChildrenOnBoard { get; set; }

    [Name("Ticket")] public string TicketSerial { get; set; }

    [Name("Fare")] public double TicketPrice { get; set; }

    public string? Cabin { get; set; }

    public string Embarked { get; set; }
}

/// <summary>
/// Predict the chance of the passenger survived on the Titanic.
/// </summary>
/// <param name="passengerClass">Passenger class.</param>
/// <param name="sex">Gender of the passenger.</param>
/// <param name="age">Age of the passenger. If the age is unknown, then it is null.</param>
/// <param name="siblingsOnBoard">Count of siblings on Titanic.</param>
/// <param name="parchOnBoard">Count of children on Titanic.</param>
/// <param name="ticketSerial">Serial number of the passenger's ticket.</param>
/// <param name="ticketPrice">Price of passenger's ticket.</param>
/// <param name="cabin">Name of the cabin that the passenger stayed.</param>
/// <param name="portOfEmbarkation">Name of the port that the passenger embarked.</param>
/// <returns>The double value of the survival chance.</returns>
public delegate double IsGoingToSurvive(int passengerClass, string sex, double? age, int siblingsOnBoard,
    int parchOnBoard, string ticketSerial, double ticketPrice, string? cabin, string portOfEmbarkation);

public class DemoTitanic : DemoBase<IsGoingToSurvive, TitanicEntry, double>
{
    public override string DemoName { get; } = "Titanic Survival Prediction";

    public override Expression<Action<IsGoingToSurvive, TitanicEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(
            entry.PassengerClass, entry.Gender, entry.Age,
            entry.SiblingOnBoard, entry.ChildrenOnBoard,
            entry.TicketSerial, entry.TicketPrice,
            entry.Cabin, entry.Embarked
        );

    public override Func<TitanicEntry, double> ResultMapping { get; }
        = entry => entry.Survived ? 1.0 : 0.0;

    public override Func<double, double, bool>? ResultVerifier { get; }
        = (expected, actual) => expected > 0.5 == actual > 0.5;

    protected override TitanicEntry[] PrepareData()
    {
        return DataLoader.FromCsv<TitanicEntry>("../Titanic.csv").ToArray();
    }

    protected override void ConfigureTrainer(Trainer<IsGoingToSurvive, TitanicEntry, double> trainer)
    {
        trainer.Metrics
            .UseCorrectness(trainer.ResultVerifier)
            .UseAccuracy(trainer.ResultVerifier);
    }
}