using System.Linq.Expressions;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Kernel;
using Robotless.Modules.AiAgent.Agents;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Robotless.Modules.OpenAi;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class DemoTitanicMixed : IDemo
{
    public string DemoName { get; } = "Titanic Survival Prediction Mixed";

    public Expression<Action<IsGoingToSurvive, TitanicEntry>> ArgumentsMapping { get; }
        = (functor, entry) => functor(
            entry.PassengerClass, entry.Gender, entry.Age,
            entry.SiblingOnBoard, entry.ChildrenOnBoard,
            entry.TicketSerial, entry.TicketPrice,
            entry.Cabin, entry.Embarked
        );

    public Func<TitanicEntry, double> ResultMapping { get; }
        = entry => entry.Survived ? 1.0 : 0.0;

    public Func<double, double, bool>? ResultVerifier { get; }
        = (expected, actual) => expected > 0.5 == actual > 0.5;

    protected TitanicEntry[] PrepareData()
    {
        return DataLoader.FromCsv<TitanicEntry>("../Titanic.csv").ToArray();
    }

    protected void ConfigureTrainer(Trainer<IsGoingToSurvive, TitanicEntry, double> trainer)
    {
        trainer.Metrics
            .UseCorrectness(trainer.ResultVerifier)
            .UseAccuracy(trainer.ResultVerifier);
    }
    
    public async Task Run(DevelopmentWorkspace workspace, TestConfiguration configuration)
    {
        var mock = workspace.NewObject<MockDelegate<IsGoingToSurvive>>();
        var logger = mock.AttachComponent<LoggerComponent>();
        logger.AuthorName = DemoName;
        mock.ReflectionAgent = new OpenAiAgent(
            Environment.GetEnvironmentVariable("OPENAI_API_KEY")!, "gpt-4o")
        {
            Workspace = workspace.RequireInjection<IWorkspace>(),
            Logger = workspace.GetInjection<LoggerComponent>(),
        };

        var data = PrepareData();
        
        var trainer = new Trainer<IsGoingToSurvive, TitanicEntry, double>(mock, data,
            ArgumentsMapping, ResultMapping, ResultVerifier)
        {
            Logger = mock.GetLogger(),
        };
        ConfigureTrainer(trainer);

        mock.Script.OnCodeChanged += code =>
        {
            trainer.Logger?.LogDetails(LogLevel.Information, "Script Changed.",
                details => { details.ScriptCode = code ?? "[Empty]"; });
        };

        var trainingCount = configuration.TrainingCount;
        var evaluationCount = data.Length - trainingCount;
        mock.EnableStructuredOutput = configuration.EnableStructuredOutput;
        mock.EnableSubstitutionScript = configuration.EnableSubstitutionScript;
        trainer.CompressionThreshold = configuration.CompressionThreshold;
        trainer.ReplacementThreshold = configuration.ReplacementThreshold;

        Console.Title = $"{DemoName} - {configuration.ModelName} - {configuration.TrainingCount}" +
                        $"{(configuration.EnableSubstitutionScript? " - Script" : "")}";
        AnsiConsole.MarkupLine($"[green]{DemoName}:[/] [blue]{configuration.ModelName}[/] - " +
                               $"{configuration.TrainingCount}" +
                               $"{(configuration.EnableSubstitutionScript? " - [green]Script[/]" : "")}");
        AnsiConsole.MarkupLine($"[bold]Session GUID:[/] [blue]{mock.Identifier}[/]");

        trainer.Logger?.LogDetails(LogLevel.Information, "Experiment Description",
            details =>
            {
                details.ModelName = configuration.ModelName;
                details.TrainingCount = trainingCount;
                details.TestCount = evaluationCount;
                details.WithScriptAcceleration = mock.EnableSubstitutionScript;
                details.CompressionThreshold = trainer.CompressionThreshold!;
                details.ReplacementThreshold = trainer.ReplacementThreshold!;
            });

        await trainer.Train(trainingCount);

        if (mock.EnableSubstitutionScript)
        {
            while (mock.Script.Code == null)
            {
                if (await mock.Script.Build())
                    break;
            }
        }
        
        await trainer.Evaluate(evaluationCount);
        
        AnsiConsole.MarkupLine($"[green]{DemoName}:[/] [blue]{configuration.ModelName}[/] - " +
                               $"{configuration.TrainingCount}" +
                               $"{(configuration.EnableSubstitutionScript? " - " +
                                                                           "[green]Script[/]" : "")}");
        AnsiConsole.MarkupLine($"[bold]Session GUID:[/] [blue]{mock.Identifier}[/]");
        
        Console.Beep();
    }
}