﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using System.Text;
using CsvHelper.Configuration.Attributes;
using JetBrains.Annotations;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.AiAgent;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class TitanicEntryRAG1
{
    public int PassengerId { get; set; }

    public bool Survived { get; set; }

    [Name("Pclass")] public int PassengerClass { get; set; }

    public string Name { get; set; }

    [Name("Sex")] public string Gender { get; set; }

    public double? Age { get; set; }

    [Name("SibSp")] public int SiblingOnBoard { get; set; }

    [Name("Parch")] public int ChildrenOnBoard { get; set; }

    [Name("Ticket")] public string TicketSerial { get; set; }

    [Name("Fare")] public double TicketPrice { get; set; }

    public string? Cabin { get; set; }

    public string Embarked { get; set; }
}

/// <summary>
/// Predict the chance of the passenger survived on the Titanic.
/// </summary>
/// <param name="passengerClass">Passenger class.</param>
/// <param name="sex">Gender of the passenger.</param>
/// <param name="age">Age of the passenger. If the age is unknown, then it is null.</param>
/// <param name="siblingsOnBoard">Count of siblings on Titanic.</param>
/// <param name="parchOnBoard">Count of children on Titanic.</param>
/// <param name="ticketSerial">Serial number of the passenger's ticket.</param>
/// <param name="ticketPrice">Price of passenger's ticket.</param>
/// <param name="cabin">Name of the cabin that the passenger stayed.</param>
/// <param name="portOfEmbarkation">Name of the port that the passenger embarked.</param>
/// <returns>The double value of the survival chance.</returns>
public delegate bool IsGoingToSurviveRAG1(int passengerClass, string sex, double? age, int siblingsOnBoard,
    int parchOnBoard, string ticketSerial, double ticketPrice, string? cabin, string portOfEmbarkation);

public class DemoTitanicRag1 : DemoBase<IsGoingToSurviveRAG1, TitanicEntryRAG1, bool>
{
    public override string DemoName { get; } = "Titanic Survival Prediction RAG1";

    public override Expression<Action<IsGoingToSurviveRAG1, TitanicEntryRAG1>> ArgumentsMapping { get; }
        = (functor, entry) => functor(
            entry.PassengerClass, entry.Gender, entry.Age,
            entry.SiblingOnBoard, entry.ChildrenOnBoard,
            entry.TicketSerial, entry.TicketPrice,
            entry.Cabin, entry.Embarked
        );

    public override Func<TitanicEntryRAG1, bool> ResultMapping { get; }
        = entry => entry.Survived;
    
    public override Func<bool, bool, bool>? ResultVerifier { get; }
        = (expected, actual) => expected == actual;
    
    protected override TitanicEntryRAG1[] PrepareData()
    {
        return DataLoader.FromCsv<TitanicEntryRAG1>("../Titanic.csv").ToArray();
    }

    protected override void ConfigureTrainer(Trainer<IsGoingToSurviveRAG1, TitanicEntryRAG1, bool> trainer)
    {
        trainer.Metrics
            .UseCorrectness(trainer.ResultVerifier)
            .UseAccuracy(trainer.ResultVerifier);
    }

    protected override void ConfigureRag(MockFunction mock)
    {
        InjectMaterialLevel2(mock);
    }
    
    public static void InjectDocuments(MockFunction mock, [LanguageInjection("markdown")] params Span<string> documents)
    {
        var builder = new StringBuilder();

        builder.AppendLine("Here are some related documents for you to reference:");
        foreach (var document in documents)
        {
            builder.AppendLine(document);
            builder.AppendLine();
        }

        mock.Memory.AddUserMessage(builder.ToString());
    }

    public static void InjectMaterialLevel0(MockFunction mock)
    {
        InjectDocuments(mock,
            """
            Titanic lacked a searchlight, 
            in accordance with the ban on the use of searchlights in the merchant navy.
            """,
            """
            The passenger facilities aboard Titanic aimed to meet the highest standards of luxury. 
            According to Titanic's general arrangement plans, 
            the ship could accommodate 833 First Class Passengers, 
            614 in Second Class and 1,006 in Third Class, for a total passenger capacity of 2,453. 
            In addition, Titanic's capacity for crew members exceeded 900, as most documents of the original 
            configuration have stated that the full carrying capacity for passengers and crew was approximately 
            3,547. The ship's interior design was a departure from that of other passenger liners, 
            which had typically been decorated in the style of a manor house or an English country house.
            """,
            """
            Titanic's maiden voyage began on Wednesday, 10 April 1912. 
            Following the embarkation of the crew, 
            the passengers began arriving at 9:30 am, when the London and South Western Railway's boat train from 
            London Waterloo station reached Southampton Terminus railway station on the quayside, 
            alongside Titanic's berth.
            The large number of Third Class passengers meant they were the first to board, 
            with First and Second Class passengers following up to an hour before departure. 
            Stewards showed them to their cabins, and First Class passengers were personally greeted by Captain Smith.
            Third Class passengers were inspected for ailments and physical impairments that might lead to their being 
            refused entry to the United States – a prospect the White Star Line wished to avoid, 
            as it would have to carry anyone who failed the examination back across the Atlantic.
            In all, 920 passengers boarded Titanic at Southampton – 179 First Class, 247 Second Class, 
            and 494 Third Class. Additional passengers were to be picked up at Cherbourg and Queenstown.
            """,
            """
            Titanic's passengers numbered approximately 1,317 people: 324 in First Class, 284 in Second Class, and 709 in Third Class. Of these, 869 (66%) were male and 447 (34%) female. There were 107 children aboard, the largest number of whom were in Third Class.[118] The ship was considerably under-capacity for the maiden voyage and could have accommodated 2,453 passengers—833 First Class, 614 Second Class, and 1,006 Third Class.[119]
            """,
            """
            A Deck, also called the promenade deck, extended along the entire 546 feet (166 m) length of the superstructure. It was reserved exclusively for First Class passengers and contained First Class cabins, the First Class reading and writing room, lounge, smoke room, and Palm Court.[18]
            B Deck, the bridge deck, was the top weight-bearing deck and the uppermost level of the hull. More First Class passenger accommodations were located here with six palatial staterooms (cabins) featuring their own private promenades. On Titanic, the à la carte restaurant and the Café Parisien provided luxury dining facilities to First Class passengers. Both were run by subcontracted chefs and their staff; all were lost in the disaster. The Second Class smoking room and entrance hall were both located on this deck. The raised forecastle of the ship was forward of the bridge deck, accommodating Number 1 hatch (the main hatch through to the cargo holds), numerous pieces of machinery and the anchor housings.[b] Aft of the bridge deck was the raised poop deck, 106 feet (32 m) long, used as a promenade by Third Class passengers. It was where many of Titanic's passengers and crew made their last stand as the ship sank. The forecastle and poop deck were separated from the bridge deck by well decks.[20][21]
            C Deck, the shelter deck, was the highest deck to run uninterrupted from stem to stern. It included both well decks; the aft one served as part of the Third-Class promenade. Crew cabins were housed below the forecastle and Third-Class public rooms were housed below the poop deck. In between were the majority of First Class cabins and the Second-Class library. [20][22]
            D Deck, the saloon deck, was dominated by three public rooms—the First-Class reception room, the First-Class dining saloon and the Second-Class dining saloon. The first- and second-class galleys were also located on this deck. An open space was provided for Third Class passengers. First, Second- and Third-Class passengers had cabins on this deck, with berths for firemen located in the bow. It was the highest level reached by the ship's watertight bulkheads (though only by eight of the fifteen bulkheads).[20][23]
            E Deck, the upper deck, was predominantly used for passenger accommodation for all three classes plus berths for cooks, seamen, stewards and trimmers. Along its length ran a long passageway nicknamed 'Scotland Road', in reference to a famous street in Liverpool. Scotland Road was used by Third Class passengers and crew members.[20][24]
            F Deck, the middle deck, mainly accommodated Second- and Third-Class passengers and several departments of the crew. The Third Class dining saloon was located here, as was the First Class bath complex, containing the swimming pool and the Turkish bath.[20][24][25]
            G Deck, the lower deck, had the lowest portholes, just above the waterline. The first-class squash court was located here along with the travelling post office where letters and parcels were sorted ready for delivery when the ship docked. Food was also stored here. The deck was interrupted at several points by orlop (partial) decks over the boiler, engine and turbine rooms.[20][26]
            The orlop decks, and the tank top below that, were on the lowest level of the ship, below the waterline. The orlop decks were used as cargo spaces, while the tank top—the inner bottom of the ship's hull—provided the platform on which the ship's boilers, engines, turbines and electrical generators were housed. This area of the ship was occupied by the engine and boiler rooms, areas which passengers would have been prohibited from seeing. They were connected with higher levels of the ship by two flights of stairs in the fireman's passage; twin spiral stairways near the bow provided access up to D Deck.[20][26] Ladders in the boiler, turbine, and engine rooms provided access to higher decks in those compartments.
            """);
    }

    public static void InjectMaterialLevel1(MockFunction mock)
    {
        InjectDocuments(mock,
            """
            Titanic was equipped with 16 lifeboat davits, each capable of lowering three lifeboats, 
            for a total capacity of 48 boats. Despite this capacity, 
            the ship was scantly equipped with a total of only 20 lifeboats. 
            Fourteen of these were regular lifeboats, two were cutter lifeboats, 
            and four were collapsible and proved difficult to launch while the ship was sinking. 
            Together, the 20 lifeboats could hold 1,178 people — roughly half the number of passengers on board, 
            and a third of the number the passengers the ship could have carried at full capacity 
            (a number consistent with the maritime safety regulations of the era). 
            The British Board of Trade's regulations required 14 lifeboats for a ship 10,000 tonnes. 
            Titanic carried six more than required, allowing 338 extra people room in lifeboats. 
            When the ship sank, the lifeboats that had been lowered were only filled up to an average of 60%.
            """,
            """
            Like Olympic, Titanic carried a total of 20 lifeboats: 
            14 standard wooden Harland and Wolff lifeboats with a capacity of 65 people each and
            four Engelhardt "collapsible" (wooden bottom, collapsible canvas sides) lifeboats 
            (identified as A to D) with a capacity of 47 people each. 
            In addition, Titanic had two emergency cutters with a capacity of 40 people each.
            Olympic carried at least two collapsible boats on either side of the number one funnel.
            All of the lifeboats were stowed securely on the boat deck and, except for collapsible lifeboats A and B, 
            connected to davits by ropes. Those on the starboard side were odd-numbered 1–15 from bow to stern, 
            while those on the port side were even-numbered 2–16 from bow to stern.
            """,
            """
            Titanic had 16 sets of davits, each able to handle three lifeboats, unlike what Carslile had hoped. 
            This gave Titanic the ability to carry up to 48 wooden lifeboats.
            However, the White Star Line decided that only 16 wooden lifeboats and four collapsibles would be carried, 
            which could accommodate 1,178 people, only one-third of Titanic's total capacity. 
            At the time, the Board of Trade's regulations required British vessels over 10,000 tonnes to only carry 16 
            lifeboats with a capacity of 990 occupants.
            """,
            """
            The boat deck, on which the lifeboats were housed. 
            It was from here during the early hours of 15 April 1912 that Titanic's lifeboats were lowered into the 
            North Atlantic. The bridge and wheelhouse were at the forward end, in front of the captain's and officers' 
            quarters. The bridge stood 8 feet (2.4 m) above the deck, extending out to either side so that the ship 
            could be controlled while docking. The wheelhouse stood within the bridge. 
            The entrance to the First Class Grand Staircase and gymnasium were located midships along with 
            the raised roof of the First Class lounge, while at the rear of the deck were the roof of the 
            First Class smoke room and the Second Class entrance. Just forward of the Second Class entrance 
            sat the kennels, where the First Class passengers' dogs would stay. 
            The wood-covered deck was divided into four segregated promenades: for officers, 
            First Class passengers, engineers, and Second Class passengers respectively. 
            Lifeboats lined the side of the deck except in the First Class area, 
            where there was a gap so that the view would not be spoiled.
            """
        );
    }

    public static void InjectMaterialLevel2(MockFunction mock)
    {
        InjectDocuments(mock,
            """
            | Sex/Age  | Class/Crew   | Number Aboard | Number Saved | Number Lost | Percentage Saved | Percentage Lost |
            | -------- | ------------ | ------------- | ------------ | ----------- | ---------------- | --------------- |
            | Children | First Class  | 6             | 5            | 1           | 83%              | 17%             |
            | Children | Second Class | 24            | 24           | 0           | 100%             | 0%              |
            | Children | Third Class  | 79            | 27           | 52          | 34%              | 66%             |
            | Women    | First Class  | 144           | 140          | 4           | 97%              | 3%              |
            | Women    | Second Class | 93            | 80           | 13          | 86%              | 14%             |
            | Women    | Third Class  | 165           | 76           | 89          | 46%              | 54%             |
            | Women    | Crew         | 23            | 20           | 3           | 87%              | 13%             |
            | Men      | First Class  | 175           | 57           | 118         | 33%              | 67%             |
            | Men      | Second Class | 168           | 14           | 154         | 8%               | 92%             |
            | Men      | Third Class  | 462           | 75           | 387         | 16%              | 84%             |
            | Men      | Crew         | 885           | 192          | 693         | 22%              | 78%             |
            | Total    |              | 2224          | 710          | 1514        | 32%              | 68%             |
            """);
    }

    public static void InjectMaterialLevel3(MockFunction mock)
    {
        InjectDocuments(mock,
            """
            Cumings, Mrs. John Bradley (Florence Briggs Thayer)
            Heikkinen, Miss. Laina
            Futrelle, Mrs. Jacques Heath (Lily May Peel)
            Johnson, Mrs. Oscar W (Elisabeth Vilhelmina Berg)
            Nasser, Mrs. Nicholas (Adele Achem)
            Sandstrom, Miss. Marguerite Rut
            Bonnell, Miss. Elizabeth
            Hewlett, Mrs. (Mary D Kingcome) 
            Williams, Mr. Charles Eugene
            Masselmani, Mrs. Fatima
            Beesley, Mr. Lawrence
            McGowan, Miss. Anna "Annie"
            Sloper, Mr. William Thompson
            Asplund, Mrs. Carl Oscar (Selma Augusta Emilia Johansson)
            O'Dwyer, Miss. Ellen "Nellie"
            Spencer, Mrs. William Augustus (Marie Eugenie)
            Glynn, Miss. Mary Agatha
            Mamee, Mr. Hanna
            Nicola-Yarred, Miss. Jamila
            Laroche, Miss. Simonne Marie Anne Andree
            Devaney, Miss. Margaret Delia
            O'Driscoll, Miss. Bridget
            Harper, Mrs. Henry Sleeper (Myna Haxtun)
            Faunthorpe, Mrs. Lizzie (Elizabeth Anne Wilkinson)
            Woolner, Mr. Hugh
            Rugg, Miss. Emily
            West, Miss. Constance Mirium
            Icard, Miss. Amelie
            Moubarek, Master. Gerios
            Nye, Mrs. (Elizabeth Ramell)
            Andersson, Miss. Erna Alexandra
            Bing, Mr. Lee
            Caldwell, Master. Alden Gates
            Dowdell, Miss. Elizabeth
            Sheerlinck, Mr. Jan Baptist
            McDermott, Miss. Brigdet Delia
            Ilett, Miss. Bertha
            Backstrom, Mrs. Karl Alfred (Maria Mathilda Gustafsson)
            Fortune, Miss. Mabel Helen
            Greenfield, Mr. William Bertram
            Doling, Mrs. John T (Ada Julia Bone)
            Salkjelsvik, Miss. Anna Kristine
            Moss, Mr. Albert Johan
            Moran, Miss. Bertha
            Webber, Miss. Susan
            Nicola-Yarred, Master. Elias
            Madsen, Mr. Fridtjof Arne
            Peter, Miss. Anna
            Weisz, Mrs. Leopold (Mathilde Francoise Pede)
            Newsom, Miss. Helen Monypeny
            Nysten, Miss. Anna Sofia
            Hakkarainen, Mrs. Pekka Pietari (Elin Matilda Dolck)
            Andersson, Mr. August Edvard ("Wennerstrom")
            Pears, Mrs. Thomas (Edith Wearne)
            Gilnagh, Miss. Katherine "Katie"
            Watt, Mrs. James (Elizabeth "Bessie" Inglis Milne)
            Goldsmith, Master. Frank John William "Frankie"
            Chibnall, Mrs. (Edith Martha Bowerman)
            Johnson, Miss. Eleanor Ileen
            Becker, Master. Richard F
            Kink-Heilmann, Miss. Luise Gretchen
            O'Brien, Mrs. Thomas (Johanna "Hannah" Godfrey)
            Romaine, Mr. Charles Hallace ("Mr C Rolmane")
            Pinsky, Mrs. (Rosa)
            Andersen-Jensen, Miss. Carla Christine Nielsine
            Navratil, Master. Michel M
            Brown, Mrs. James Joseph (Margaret Tobin)
            Lurette, Miss. Elise
            Madigan, Miss. Margaret "Maggie"
            Cohen, Mr. Gurshon "Gus"
            Albimona, Mr. Nassef Cassem
            Carr, Miss. Helen "Ellen"
            Blank, Mr. Henry
            Cameron, Miss. Clear Annie
            Newell, Miss. Madeleine
            Honkanen, Miss. Eliina
            Bazzani, Miss. Albina
            Sunderland, Mr. Victor Francis
            Hoyt, Mr. Frederick Maxfield
            Mellors, Mr. William John
            Harris, Mrs. Henry Birkhardt (Irene Wallach)
            Asplund, Miss. Lillian Gertrud
            Collyer, Miss. Marjorie "Lottie"
            Murphy, Miss. Katherine "Kate"
            Hamalainen, Mrs. William (Anna)
            Beckwith, Mr. Richard Leonard
            Touma, Mrs. Darwis (Hanne Youssef Razi)
            Thorne, Mrs. Gertrude Maybelle
            Cherry, Miss. Gladys
            Ward, Miss. Anna
            Parrish, Mrs. (Lutie Davis)
            Asplund, Master. Edvin Rojj Felix
            Persson, Mr. Ernst Ulrik
            Graham, Mrs. William Thompson (Edith Junkins)
            Bissette, Miss. Amelia
            Tornquist, Mr. William Henry
            Mellinger, Mrs. (Elizabeth Anne Maidment)
            Healy, Miss. Hanora "Nora"
            Andrews, Miss. Kornelia Theodosia
            Abbott, Mrs. Stanton (Rosa Hunt)
            Dorking, Mr. Edward Arthur
            de Mulder, Mr. Theodore
            Hosono, Mr. Masabumi
            Connolly, Miss. Kate
            Barber, Miss. Ellen "Nellie"
            Bishop, Mrs. Dickinson H (Helen Walton)
            Saalfeld, Mr. Adolphe
            Baxter, Mrs. James (Helene DeLaudeniere Chaput)
            Kelly, Miss. Anna Katherine "Annie Kate"
            McCoy, Mr. Bernard
            Keane, Miss. Nora A
            Allison, Master. Hudson Trevor
            Fleming, Miss. Margaret
            Penasco y Castellana, Mrs. Victor de Satode (Maria Josefa Perez de Soto y Vallejo)
            Francatelli, Miss. Laura Mabel
            Hays, Miss. Margaret Bechstein
            Ryerson, Miss. Emily Borie
            Nilsson, Miss. Helmina Josefina
            Kantor, Mrs. Sinai (Miriam Sternin)
            Wick, Miss. Mary Natalie
            Spedden, Mrs. Frederic Oakley (Margaretta Corning Stone)
            Slayter, Miss. Hilda Mary
            Caldwell, Mrs. Albert Francis (Sylvia Mae Harbaugh)
            Young, Miss. Marie Grice
            Ball, Mrs. (Ada E Hall)
            Goldsmith, Mrs. Frank John (Emily Alice Brown)
            Hippach, Miss. Jean Gertrude
            McCoy, Miss. Agnes
            Frauenthal, Mrs. Henry William (Clara Heinsheimer)
            Burns, Miss. Elizabeth Margaret
            Dahl, Mr. Karl Edwart
            Navratil, Master. Edmond Roger
            Fortune, Miss. Alice Elizabeth
            Brown, Miss. Amelia "Mildred"
            Smith, Miss. Marion Elsie
            Davison, Mrs. Thomas Henry (Mary E Finck)
            Coutts, Master. William Loch "William"
            Bowerman, Miss. Elsie Edith
            McGovern, Miss. Mary
            Mockler, Miss. Helen Mary "Ellie"
            Warren, Mrs. Frank Manley (Anna Sophia Atkinson)
            Moussa, Mrs. (Mantoura Boulos)
            Jermyn, Miss. Annie
            Aubart, Mme. Leontine Pauline
            Harder, Mr. George Achilles
            Meyer, Mrs. Edgar Joseph (Leila Saks)
            Landergren, Miss. Aurora Adelia
            Bidois, Miss. Rosalie
            Nakid, Miss. Maria ("Mary")
            Holverson, Mrs. Alexander Oskar (Mary Aline Towner)
            Buss, Miss. Kate
            Lehmann, Miss. Bertha
            Carter, Mr. William Ernest
            Jansson, Mr. Carl Olof
            Newell, Miss. Marjorie
            Sandstrom, Mrs. Hjalmar (Agnes Charlotta Bengtsson)
            Trout, Mrs. William H (Jessie L)
            Niskanen, Mr. Juha
            Richards, Master. William Rowe
            Minahan, Miss. Daisy E
            Sundman, Mr. Johan Julian
            Drew, Mrs. James Vivian (Lulu Thorne Christian)
            Silven, Miss. Lyyli Karoliina
            Clarke, Mrs. Charles V (Ada Maria Winfield)
            Phillips, Miss. Kate Florence ("Mrs Kate Louise Phillips Marshall")
            Pickard, Mr. Berk (Berk Trembisky)
            Bjornstrom-Steffansson, Mr. Mauritz Hakan
            Thorneycroft, Mrs. Percival (Florence Kate White)
            Louch, Mrs. Charles Alexander (Alice Adelaide Slow)
            Carter, Miss. Lucile Polk
            Richards, Mrs. Sidney (Emily Hocking)
            Hart, Mrs. Benjamin (Esther Ada Bloomfield)
            Reynaldo, Ms. Encarnacion
            Johannesen-Bratthammer, Mr. Bernt
            Dodge, Master. Washington
            Mellinger, Miss. Madeleine Violet
            Seward, Mr. Frederic Kimber
            Baclini, Miss. Marie Catherine
            Peuchen, Major. Arthur Godfrey
            Goldenberg, Mr. Samuel L
            Jalsevac, Mr. Ivan
            Kenyon, Mrs. Frederick R (Marion)
            Toomey, Miss. Ellen
            Anderson, Mr. Harry
            Baclini, Miss. Helene Barbara
            West, Mrs. Edwy Arthur (Ada Mary Worth)
            Jerwan, Mrs. Amin S (Marie Marthe Thuillard)
            Hirvonen, Miss. Hildur E
            Turkula, Mrs. (Hedwig)
            Bishop, Mr. Dickinson H
            Hoyt, Mrs. Frederick Maxfield (Jane Anne Forby)
            Coutts, Master. Eden Leslie "Neville"
            Eustis, Miss. Elizabeth Mussey
            Maioni, Miss. Roberta
            Quick, Mrs. Frederick Charles (Jane Richards)
            Bradley, Mr. George ("George Arthur Brayton")
            Lang, Mr. Fang
            Daly, Mr. Eugene Patrick
            McGough, Mr. James Robert
            Rothschild, Mrs. Martin (Elizabeth L. Barrett)
            Lemore, Mrs. (Amelia Milley)
            Angle, Mrs. William A (Florence "Mary" Agnes Hughes)
            Perreault, Miss. Anne
            Hippach, Mrs. Louis Albert (Ida Sophia Fischer)
            Ridsdale, Miss. Lucy
            Quick, Miss. Phyllis May
            Peter, Mrs. Catherine (Catherine Rizk)
            Hart, Miss. Eva Miriam
            LeRoy, Miss. Bertha
            Frolicher, Miss. Hedwig Margaritha
            Crosby, Miss. Harriet R
            Beane, Mr. Edward
            Beane, Mrs. Edward (Ethel Clarke)
            Padro y Manent, Mr. Julian
            Davies, Master. John Morgan Jr
            Thayer, Mr. John Borland Jr
            Leeni, Mr. Fahim ("Philip Zenni")
            Ohman, Miss. Velin
            Duff Gordon, Lady. (Lucille Christiana Sutherland) ("Mrs Morgan")
            Taussig, Mrs. Emil (Tillie Mandelbaum)
            de Messemaeker, Mrs. Guillaume Joseph (Emma)
            Jonsson, Mr. Carl
            Harris, Mr. George
            Appleton, Mrs. Edward Dale (Charlotte Lamson)
            Flynn, Mr. John Irwin ("Irving")
            Kelly, Miss. Mary
            Garside, Miss. Ethel
            Silvey, Mrs. William Baird (Alice Munger)
            Jussila, Mr. Eiriik
            Christy, Miss. Julie Rachel
            Thayer, Mrs. John Borland (Marian Longstreth Morris)
            Taussig, Miss. Ruth
            Frolicher-Stehli, Mr. Maxmillian
            Stephenson, Mrs. Walter Bertram (Martha Eustis)
            Leitch, Miss. Jessie Wills
            Duff Gordon, Sir. Cosmo Edmund ("Mr Morgan")
            Jacobsohn, Mrs. Sidney Samuel (Amy Frances Christy)
            Homer, Mr. Harry ("Mr E Haven")
            Daniel, Mr. Robert Williams
            Laroche, Mrs. Joseph (Juliette Marie Louise Lafargue)
            Shutes, Miss. Elizabeth W
            Murphy, Miss. Margaret Jane
            Herman, Miss. Alice
            Becker, Miss. Marion Louise
            Kimball, Mr. Edwin Nelson Jr
            Nakid, Mr. Sahid
            Longley, Miss. Gretchen Fiske
            Barkworth, Mr. Algernon Henry Wilson
            Stahelin-Maeglin, Dr. Max
            Davis, Miss. Mary
            Sagesser, Mlle. Emma
            Foo, Mr. Choong
            Baclini, Miss. Eugenie
            Harper, Mr. Henry Sleeper
            Simonius-Blumer, Col. Oberst Alfons
            Stanley, Miss. Amy Zillah Elsie
            Doling, Miss. Elsie
            O'Leary, Miss. Hanora "Norah"
            Frauenthal, Dr. Henry William
            Lindqvist, Mr. Eino William
            Taylor, Mrs. Elmer Zebley (Juliet Cummins Wright)
            Brown, Mrs. Thomas William Solomon (Elizabeth Catherine Ford)
            Wilhelms, Mr. Charles
            Turja, Miss. Anna Sofia
            Cardeza, Mr. Thomas Drake Martinez
            Hassab, Mr. Hammad
            Madill, Miss. Georgette Alexandra
            Dick, Mr. Albert Adrian
            Karun, Miss. Manca
            Lam, Mr. Ali
            Mullens, Miss. Katherine "Katie"
            Astor, Mrs. John Jacob (Madeleine Talmadge Force)
            Silverthorne, Mr. Spencer Victor
            Kelly, Mrs. Florence "Fannie"
            Calderhead, Mr. Edward Pennington
            Cleaver, Miss. Alice
            Moubarek, Master. Halim Gonios ("William George")
            Mayne, Mlle. Berthe Antonine ("Mrs de Villiers")
            Taylor, Mr. Elmer Zebley
            Endres, Miss. Caroline Louise
            Troutt, Miss. Edwina Celia "Winnie"
            Harper, Miss. Annie Jessie "Nina"
            Chambers, Mr. Norman Campbell
            Renouf, Mrs. Peter Henry (Lillian Jefferys)
            Mannion, Miss. Margareth
            Allen, Miss. Elisabeth Walton
            Lesurer, Mr. Gustave J
            Hawksford, Mr. Walter James
            Ryerson, Miss. Susan Parker "Suzette"
            Stranden, Mr. Juho
            Sinkkonen, Miss. Anna
            Wells, Miss. Joan
            Moor, Master. Meier
            Herman, Mrs. Samuel (Jane Laver)
            Hamalainen, Master. Viljo
            Rothes, the Countess. of (Lucy Noel Martha Dyer-Edwards)
            Barah, Mr. Hanna Assi
            Carter, Mrs. William Ernest (Lucile Polk)
            Hogeboom, Mrs. John C (Anna Andrews)
            Hocking, Mrs. Elizabeth (Eliza Needs)
            Emanuel, Miss. Virginia Ethel
            Robert, Mrs. Edward Scott (Elisabeth Walton McMillan)
            Ayoub, Miss. Banoura
            Dick, Mrs. Albert Adrian (Vera Gillespie)
            Sjoblom, Miss. Anna Sofia
            Dean, Master. Bertram Vere
            Leader, Dr. Alice (Farnham)
            Osman, Mrs. Mara
            Collyer, Mrs. Harvey (Charlotte Annie Tate)
            Carter, Master. William Thornton II
            Thomas, Master. Assad Alexander
            Hedman, Mr. Oskar Arvid
            Chambers, Mrs. Norman Campbell (Bertha Griggs)
            Hays, Mrs. Charles Melville (Clara Jennings Gregg)
            Lulic, Mr. Nikola
            Moor, Mrs. (Beila)
            Mallet, Master. Andre
            McCormack, Mr. Thomas Joseph
            Stone, Mrs. George Nelson (Martha Evelyn)
            Yasbeck, Mrs. Antoni (Selini Alexander)
            Richards, Master. George Sibley
            Compton, Miss. Sara Rebecca
            Chip, Mr. Chang
            Marechal, Mr. Pierre
            Serepeca, Miss. Augusta
            Goldenberg, Mrs. Samuel L (Edwiga Grabowska)
            Lines, Miss. Mary Conover
            Aks, Mrs. Sam (Leah Rosen)
            Wick, Mrs. George Dennick (Mary Hitchcock)
            Daly, Mr. Peter Denis 
            Baclini, Mrs. Solomon (Latifa Qurban)
            Swift, Mrs. Frederick Joel (Margaret Welles Barron)
            Bystrom, Mrs. (Karolina)
            Duran y More, Miss. Asuncion
            Johnson, Master. Harold Theodor
            Beckwith, Mrs. Richard Leonard (Sallie Monypeny)
            Abelson, Mrs. Samuel (Hannah Wizosky)
            Najib, Miss. Adele Kiamie "Jane"
            Potter, Mrs. Thomas Jr (Lily Alexenia Wilson)
            Shelley, Mrs. William (Imanita Parrish Hall)
            Graham, Miss. Margaret Edith
            Behr, Mr. Karl Howell
            """);
    }
}