﻿using System.Linq.Expressions;
using System.Runtime.InteropServices;
using System.Text;
using CsvHelper.Configuration.Attributes;
using JetBrains.Annotations;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.AiAgent;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Mocking.Learning;
using Robotless.Modules.Mocking.Learning.Metrics;
using Spectre.Console;

namespace Robotless.Modules.Mocking.Demo.Demos;

public class TitanicEntryRAG2
{
    public int PassengerId { get; set; }

    public bool Survived { get; set; }

    [Name("Pclass")] public int PassengerClass { get; set; }

    public string Name { get; set; }

    [Name("Sex")] public string Gender { get; set; }

    public double? Age { get; set; }

    [Name("SibSp")] public int SiblingOnBoard { get; set; }

    [Name("Parch")] public int ChildrenOnBoard { get; set; }

    [Name("Ticket")] public string TicketSerial { get; set; }

    [Name("Fare")] public double TicketPrice { get; set; }

    public string? Cabin { get; set; }

    public string Embarked { get; set; }
}

/// <summary>
/// Predict the chance of the passenger survived on the Titanic.
/// </summary>
/// <param name="passengerClass">Passenger class.</param>
/// <param name="sex">Gender of the passenger.</param>
/// <param name="age">Age of the passenger. If the age is unknown, then it is null.</param>
/// <param name="siblingsOnBoard">Count of siblings on Titanic.</param>
/// <param name="parchOnBoard">Count of children on Titanic.</param>
/// <param name="ticketSerial">Serial number of the passenger's ticket.</param>
/// <param name="ticketPrice">Price of passenger's ticket.</param>
/// <param name="cabin">Name of the cabin that the passenger stayed.</param>
/// <param name="portOfEmbarkation">Name of the port that the passenger embarked.</param>
/// <returns>The double value of the survival chance.</returns>
public delegate bool IsGoingToSurviveRAG2(int passengerClass, string name, string sex, double? age, int siblingsOnBoard,
    int parchOnBoard, string ticketSerial, double ticketPrice, string? cabin, string portOfEmbarkation);

public class DemoTitanicRag2 : DemoBase<IsGoingToSurviveRAG2, TitanicEntryRAG2, bool>
{
    public override string DemoName { get; } = "Titanic Survival Prediction RAG2";

    public override Expression<Action<IsGoingToSurviveRAG2, TitanicEntryRAG2>> ArgumentsMapping { get; }
        = (functor, entry) => functor(
            entry.PassengerClass, entry.Name, entry.Gender, entry.Age,
            entry.SiblingOnBoard, entry.ChildrenOnBoard,
            entry.TicketSerial, entry.TicketPrice,
            entry.Cabin, entry.Embarked
        );

    public override Func<TitanicEntryRAG2, bool> ResultMapping { get; }
        = entry => entry.Survived;
    
    public override Func<bool, bool, bool>? ResultVerifier { get; }
        = (expected, actual) => expected == actual;
    
    protected override TitanicEntryRAG2[] PrepareData()
    {
        return DataLoader.FromCsv<TitanicEntryRAG2>("../Titanic.csv").ToArray();
    }

    protected override void ConfigureTrainer(Trainer<IsGoingToSurviveRAG2, TitanicEntryRAG2, bool> trainer)
    {
        trainer.Metrics
            .UseCorrectness(trainer.ResultVerifier)
            .UseAccuracy(trainer.ResultVerifier);
    }

    protected override void ConfigureRag(MockFunction mock)
    {
        InjectMaterialLevel2(mock);
    }
    
    public static void InjectDocuments(MockFunction mock, [LanguageInjection("markdown")] params Span<string> documents)
    {
        var builder = new StringBuilder();

        builder.AppendLine("Here are some related documents for you to reference:");
        foreach (var document in documents)
        {
            builder.AppendLine(document);
            builder.AppendLine();
        }

        mock.Memory.AddUserMessage(builder.ToString());
    }

    public static void InjectMaterialLevel2(MockFunction mock)
    {
        InjectDocuments(mock,
            """
            Here is the list of survivors:
            Cumings, Mrs. John Bradley (Florence Briggs Thayer)
            Heikkinen, Miss. Laina
            Futrelle, Mrs. Jacques Heath (Lily May Peel)
            Johnson, Mrs. Oscar W (Elisabeth Vilhelmina Berg)
            Nasser, Mrs. Nicholas (Adele Achem)
            Sandstrom, Miss. Marguerite Rut
            Bonnell, Miss. Elizabeth
            Hewlett, Mrs. (Mary D Kingcome) 
            Williams, Mr. Charles Eugene
            Masselmani, Mrs. Fatima
            Beesley, Mr. Lawrence
            McGowan, Miss. Anna "Annie"
            Sloper, Mr. William Thompson
            Asplund, Mrs. Carl Oscar (Selma Augusta Emilia Johansson)
            O'Dwyer, Miss. Ellen "Nellie"
            Spencer, Mrs. William Augustus (Marie Eugenie)
            Glynn, Miss. Mary Agatha
            Mamee, Mr. Hanna
            Nicola-Yarred, Miss. Jamila
            Laroche, Miss. Simonne Marie Anne Andree
            Devaney, Miss. Margaret Delia
            O'Driscoll, Miss. Bridget
            Harper, Mrs. Henry Sleeper (Myna Haxtun)
            Faunthorpe, Mrs. Lizzie (Elizabeth Anne Wilkinson)
            Woolner, Mr. Hugh
            Rugg, Miss. Emily
            West, Miss. Constance Mirium
            Icard, Miss. Amelie
            Moubarek, Master. Gerios
            Nye, Mrs. (Elizabeth Ramell)
            Andersson, Miss. Erna Alexandra
            Bing, Mr. Lee
            Caldwell, Master. Alden Gates
            Dowdell, Miss. Elizabeth
            Sheerlinck, Mr. Jan Baptist
            McDermott, Miss. Brigdet Delia
            Ilett, Miss. Bertha
            Backstrom, Mrs. Karl Alfred (Maria Mathilda Gustafsson)
            Fortune, Miss. Mabel Helen
            Greenfield, Mr. William Bertram
            Doling, Mrs. John T (Ada Julia Bone)
            Salkjelsvik, Miss. Anna Kristine
            Moss, Mr. Albert Johan
            Moran, Miss. Bertha
            Webber, Miss. Susan
            Nicola-Yarred, Master. Elias
            Madsen, Mr. Fridtjof Arne
            Peter, Miss. Anna
            Weisz, Mrs. Leopold (Mathilde Francoise Pede)
            Newsom, Miss. Helen Monypeny
            Nysten, Miss. Anna Sofia
            Hakkarainen, Mrs. Pekka Pietari (Elin Matilda Dolck)
            Andersson, Mr. August Edvard ("Wennerstrom")
            Pears, Mrs. Thomas (Edith Wearne)
            Gilnagh, Miss. Katherine "Katie"
            Watt, Mrs. James (Elizabeth "Bessie" Inglis Milne)
            Goldsmith, Master. Frank John William "Frankie"
            Chibnall, Mrs. (Edith Martha Bowerman)
            Johnson, Miss. Eleanor Ileen
            Becker, Master. Richard F
            Kink-Heilmann, Miss. Luise Gretchen
            O'Brien, Mrs. Thomas (Johanna "Hannah" Godfrey)
            Romaine, Mr. Charles Hallace ("Mr C Rolmane")
            Pinsky, Mrs. (Rosa)
            Andersen-Jensen, Miss. Carla Christine Nielsine
            Navratil, Master. Michel M
            Brown, Mrs. James Joseph (Margaret Tobin)
            Lurette, Miss. Elise
            Madigan, Miss. Margaret "Maggie"
            Cohen, Mr. Gurshon "Gus"
            Albimona, Mr. Nassef Cassem
            Carr, Miss. Helen "Ellen"
            Blank, Mr. Henry
            Cameron, Miss. Clear Annie
            Newell, Miss. Madeleine
            Honkanen, Miss. Eliina
            Bazzani, Miss. Albina
            Sunderland, Mr. Victor Francis
            Hoyt, Mr. Frederick Maxfield
            Mellors, Mr. William John
            Harris, Mrs. Henry Birkhardt (Irene Wallach)
            Asplund, Miss. Lillian Gertrud
            Collyer, Miss. Marjorie "Lottie"
            Murphy, Miss. Katherine "Kate"
            Hamalainen, Mrs. William (Anna)
            Beckwith, Mr. Richard Leonard
            Touma, Mrs. Darwis (Hanne Youssef Razi)
            Thorne, Mrs. Gertrude Maybelle
            Cherry, Miss. Gladys
            Ward, Miss. Anna
            Parrish, Mrs. (Lutie Davis)
            Asplund, Master. Edvin Rojj Felix
            Persson, Mr. Ernst Ulrik
            Graham, Mrs. William Thompson (Edith Junkins)
            Bissette, Miss. Amelia
            Tornquist, Mr. William Henry
            Mellinger, Mrs. (Elizabeth Anne Maidment)
            Healy, Miss. Hanora "Nora"
            Andrews, Miss. Kornelia Theodosia
            Abbott, Mrs. Stanton (Rosa Hunt)
            Dorking, Mr. Edward Arthur
            de Mulder, Mr. Theodore
            Hosono, Mr. Masabumi
            Connolly, Miss. Kate
            Barber, Miss. Ellen "Nellie"
            Bishop, Mrs. Dickinson H (Helen Walton)
            Saalfeld, Mr. Adolphe
            Baxter, Mrs. James (Helene DeLaudeniere Chaput)
            Kelly, Miss. Anna Katherine "Annie Kate"
            McCoy, Mr. Bernard
            Keane, Miss. Nora A
            Allison, Master. Hudson Trevor
            Fleming, Miss. Margaret
            Penasco y Castellana, Mrs. Victor de Satode (Maria Josefa Perez de Soto y Vallejo)
            Francatelli, Miss. Laura Mabel
            Hays, Miss. Margaret Bechstein
            Ryerson, Miss. Emily Borie
            Nilsson, Miss. Helmina Josefina
            Kantor, Mrs. Sinai (Miriam Sternin)
            Wick, Miss. Mary Natalie
            Spedden, Mrs. Frederic Oakley (Margaretta Corning Stone)
            Slayter, Miss. Hilda Mary
            Caldwell, Mrs. Albert Francis (Sylvia Mae Harbaugh)
            Young, Miss. Marie Grice
            Ball, Mrs. (Ada E Hall)
            Goldsmith, Mrs. Frank John (Emily Alice Brown)
            Hippach, Miss. Jean Gertrude
            McCoy, Miss. Agnes
            Frauenthal, Mrs. Henry William (Clara Heinsheimer)
            Burns, Miss. Elizabeth Margaret
            Dahl, Mr. Karl Edwart
            Navratil, Master. Edmond Roger
            Fortune, Miss. Alice Elizabeth
            Brown, Miss. Amelia "Mildred"
            Smith, Miss. Marion Elsie
            Davison, Mrs. Thomas Henry (Mary E Finck)
            Coutts, Master. William Loch "William"
            Bowerman, Miss. Elsie Edith
            McGovern, Miss. Mary
            Mockler, Miss. Helen Mary "Ellie"
            Warren, Mrs. Frank Manley (Anna Sophia Atkinson)
            Moussa, Mrs. (Mantoura Boulos)
            Jermyn, Miss. Annie
            Aubart, Mme. Leontine Pauline
            Harder, Mr. George Achilles
            Meyer, Mrs. Edgar Joseph (Leila Saks)
            Landergren, Miss. Aurora Adelia
            Bidois, Miss. Rosalie
            Nakid, Miss. Maria ("Mary")
            Holverson, Mrs. Alexander Oskar (Mary Aline Towner)
            Buss, Miss. Kate
            Lehmann, Miss. Bertha
            Carter, Mr. William Ernest
            Jansson, Mr. Carl Olof
            Newell, Miss. Marjorie
            Sandstrom, Mrs. Hjalmar (Agnes Charlotta Bengtsson)
            Trout, Mrs. William H (Jessie L)
            Niskanen, Mr. Juha
            Richards, Master. William Rowe
            Minahan, Miss. Daisy E
            Sundman, Mr. Johan Julian
            Drew, Mrs. James Vivian (Lulu Thorne Christian)
            Silven, Miss. Lyyli Karoliina
            Clarke, Mrs. Charles V (Ada Maria Winfield)
            Phillips, Miss. Kate Florence ("Mrs Kate Louise Phillips Marshall")
            Pickard, Mr. Berk (Berk Trembisky)
            Bjornstrom-Steffansson, Mr. Mauritz Hakan
            Thorneycroft, Mrs. Percival (Florence Kate White)
            Louch, Mrs. Charles Alexander (Alice Adelaide Slow)
            Carter, Miss. Lucile Polk
            Richards, Mrs. Sidney (Emily Hocking)
            Hart, Mrs. Benjamin (Esther Ada Bloomfield)
            Reynaldo, Ms. Encarnacion
            Johannesen-Bratthammer, Mr. Bernt
            Dodge, Master. Washington
            Mellinger, Miss. Madeleine Violet
            Seward, Mr. Frederic Kimber
            Baclini, Miss. Marie Catherine
            Peuchen, Major. Arthur Godfrey
            Goldenberg, Mr. Samuel L
            Jalsevac, Mr. Ivan
            Kenyon, Mrs. Frederick R (Marion)
            Toomey, Miss. Ellen
            Anderson, Mr. Harry
            Baclini, Miss. Helene Barbara
            West, Mrs. Edwy Arthur (Ada Mary Worth)
            Jerwan, Mrs. Amin S (Marie Marthe Thuillard)
            Hirvonen, Miss. Hildur E
            Turkula, Mrs. (Hedwig)
            Bishop, Mr. Dickinson H
            Hoyt, Mrs. Frederick Maxfield (Jane Anne Forby)
            Coutts, Master. Eden Leslie "Neville"
            Eustis, Miss. Elizabeth Mussey
            Maioni, Miss. Roberta
            Quick, Mrs. Frederick Charles (Jane Richards)
            Bradley, Mr. George ("George Arthur Brayton")
            Lang, Mr. Fang
            Daly, Mr. Eugene Patrick
            McGough, Mr. James Robert
            Rothschild, Mrs. Martin (Elizabeth L. Barrett)
            Lemore, Mrs. (Amelia Milley)
            Angle, Mrs. William A (Florence "Mary" Agnes Hughes)
            Perreault, Miss. Anne
            Hippach, Mrs. Louis Albert (Ida Sophia Fischer)
            Ridsdale, Miss. Lucy
            Quick, Miss. Phyllis May
            Peter, Mrs. Catherine (Catherine Rizk)
            Hart, Miss. Eva Miriam
            LeRoy, Miss. Bertha
            Frolicher, Miss. Hedwig Margaritha
            Crosby, Miss. Harriet R
            Beane, Mr. Edward
            Beane, Mrs. Edward (Ethel Clarke)
            Padro y Manent, Mr. Julian
            Davies, Master. John Morgan Jr
            Thayer, Mr. John Borland Jr
            Leeni, Mr. Fahim ("Philip Zenni")
            Ohman, Miss. Velin
            Duff Gordon, Lady. (Lucille Christiana Sutherland) ("Mrs Morgan")
            Taussig, Mrs. Emil (Tillie Mandelbaum)
            de Messemaeker, Mrs. Guillaume Joseph (Emma)
            Jonsson, Mr. Carl
            Harris, Mr. George
            Appleton, Mrs. Edward Dale (Charlotte Lamson)
            Flynn, Mr. John Irwin ("Irving")
            Kelly, Miss. Mary
            Garside, Miss. Ethel
            Silvey, Mrs. William Baird (Alice Munger)
            Jussila, Mr. Eiriik
            Christy, Miss. Julie Rachel
            Thayer, Mrs. John Borland (Marian Longstreth Morris)
            Taussig, Miss. Ruth
            Frolicher-Stehli, Mr. Maxmillian
            Stephenson, Mrs. Walter Bertram (Martha Eustis)
            Leitch, Miss. Jessie Wills
            Duff Gordon, Sir. Cosmo Edmund ("Mr Morgan")
            Jacobsohn, Mrs. Sidney Samuel (Amy Frances Christy)
            Homer, Mr. Harry ("Mr E Haven")
            Daniel, Mr. Robert Williams
            Laroche, Mrs. Joseph (Juliette Marie Louise Lafargue)
            Shutes, Miss. Elizabeth W
            Murphy, Miss. Margaret Jane
            Herman, Miss. Alice
            Becker, Miss. Marion Louise
            Kimball, Mr. Edwin Nelson Jr
            Nakid, Mr. Sahid
            Longley, Miss. Gretchen Fiske
            Barkworth, Mr. Algernon Henry Wilson
            Stahelin-Maeglin, Dr. Max
            Davis, Miss. Mary
            Sagesser, Mlle. Emma
            Foo, Mr. Choong
            Baclini, Miss. Eugenie
            Harper, Mr. Henry Sleeper
            Simonius-Blumer, Col. Oberst Alfons
            Stanley, Miss. Amy Zillah Elsie
            Doling, Miss. Elsie
            O'Leary, Miss. Hanora "Norah"
            Frauenthal, Dr. Henry William
            Lindqvist, Mr. Eino William
            Taylor, Mrs. Elmer Zebley (Juliet Cummins Wright)
            Brown, Mrs. Thomas William Solomon (Elizabeth Catherine Ford)
            Wilhelms, Mr. Charles
            Turja, Miss. Anna Sofia
            Cardeza, Mr. Thomas Drake Martinez
            Hassab, Mr. Hammad
            Madill, Miss. Georgette Alexandra
            Dick, Mr. Albert Adrian
            Karun, Miss. Manca
            Lam, Mr. Ali
            Mullens, Miss. Katherine "Katie"
            Astor, Mrs. John Jacob (Madeleine Talmadge Force)
            Silverthorne, Mr. Spencer Victor
            Kelly, Mrs. Florence "Fannie"
            Calderhead, Mr. Edward Pennington
            Cleaver, Miss. Alice
            Moubarek, Master. Halim Gonios ("William George")
            Mayne, Mlle. Berthe Antonine ("Mrs de Villiers")
            Taylor, Mr. Elmer Zebley
            Endres, Miss. Caroline Louise
            Troutt, Miss. Edwina Celia "Winnie"
            Harper, Miss. Annie Jessie "Nina"
            Chambers, Mr. Norman Campbell
            Renouf, Mrs. Peter Henry (Lillian Jefferys)
            Mannion, Miss. Margareth
            Allen, Miss. Elisabeth Walton
            Lesurer, Mr. Gustave J
            Hawksford, Mr. Walter James
            Ryerson, Miss. Susan Parker "Suzette"
            Stranden, Mr. Juho
            Sinkkonen, Miss. Anna
            Wells, Miss. Joan
            Moor, Master. Meier
            Herman, Mrs. Samuel (Jane Laver)
            Hamalainen, Master. Viljo
            Rothes, the Countess. of (Lucy Noel Martha Dyer-Edwards)
            Barah, Mr. Hanna Assi
            Carter, Mrs. William Ernest (Lucile Polk)
            Hogeboom, Mrs. John C (Anna Andrews)
            Hocking, Mrs. Elizabeth (Eliza Needs)
            Emanuel, Miss. Virginia Ethel
            Robert, Mrs. Edward Scott (Elisabeth Walton McMillan)
            Ayoub, Miss. Banoura
            Dick, Mrs. Albert Adrian (Vera Gillespie)
            Sjoblom, Miss. Anna Sofia
            Dean, Master. Bertram Vere
            Leader, Dr. Alice (Farnham)
            Osman, Mrs. Mara
            Collyer, Mrs. Harvey (Charlotte Annie Tate)
            Carter, Master. William Thornton II
            Thomas, Master. Assad Alexander
            Hedman, Mr. Oskar Arvid
            Chambers, Mrs. Norman Campbell (Bertha Griggs)
            Hays, Mrs. Charles Melville (Clara Jennings Gregg)
            Lulic, Mr. Nikola
            Moor, Mrs. (Beila)
            Mallet, Master. Andre
            McCormack, Mr. Thomas Joseph
            Stone, Mrs. George Nelson (Martha Evelyn)
            Yasbeck, Mrs. Antoni (Selini Alexander)
            Richards, Master. George Sibley
            Compton, Miss. Sara Rebecca
            Chip, Mr. Chang
            Marechal, Mr. Pierre
            Serepeca, Miss. Augusta
            Goldenberg, Mrs. Samuel L (Edwiga Grabowska)
            Lines, Miss. Mary Conover
            Aks, Mrs. Sam (Leah Rosen)
            Wick, Mrs. George Dennick (Mary Hitchcock)
            Daly, Mr. Peter Denis 
            Baclini, Mrs. Solomon (Latifa Qurban)
            Swift, Mrs. Frederick Joel (Margaret Welles Barron)
            Bystrom, Mrs. (Karolina)
            Duran y More, Miss. Asuncion
            Johnson, Master. Harold Theodor
            Beckwith, Mrs. Richard Leonard (Sallie Monypeny)
            Abelson, Mrs. Samuel (Hannah Wizosky)
            Najib, Miss. Adele Kiamie "Jane"
            Potter, Mrs. Thomas Jr (Lily Alexenia Wilson)
            Shelley, Mrs. William (Imanita Parrish Hall)
            Graham, Miss. Margaret Edith
            Behr, Mr. Karl Howell
            """);
    }
}