﻿using System.ClientModel;
using System.Reflection;
using Microsoft.Extensions.Logging;
using MongoDB.Bson;
using MongoDB.Driver;
using Robotless.Modules.AiAgent;
using Robotless.Modules.AiAgent.Agents;
using Robotless.Modules.Developing;
using Robotless.Modules.Documenting;
using Robotless.Modules.Documenting.Xml;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Logging.Handlers;
using Robotless.Modules.Mocking.Demo.Demos;
using Robotless.Modules.Mocking.Demo.Providers;
using Robotless.Modules.Mocking.Demo.Utilities;
using Robotless.Modules.Serializing;
using Robotless.Modules.Serializing.Serializers.BsonValues;
using Robotless.Modules.Serializing.Serializers.Containers;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Spectre.Console;

#pragma warning disable CS1591 // Missing XML comment for publicly visible type or member

namespace Robotless.Modules.Mocking.Demo;

public static class Launcher
{
    public static async Task Main(string[] arguments)
    {
        var client = new MongoClient(Environment.GetEnvironmentVariable("MONGODB_URL")!);
        var database = client.GetDatabase("Robotless");

        var workspace = new DevelopmentWorkspace();
        workspace.Injections.AddConstant(client);
        workspace.Injections.AddConstant(database);
        workspace.Injections.AddXmlDocumentation(typeof(Launcher).Assembly);
        workspace.Injections.AddSerialization((context, provider) =>
        {
            context
                .WithPrimitiveSerializers()
                .WithContainerSerializers()
                .WithBsonValueSerializers()
                .WithGenerator(provider.GetInjection<IDocumentation>());
        });

        var logCenter = (LogCenter)workspace.InstallWorkspaceService(typeof(LogCenter));

        logCenter.UseLogHandler(new DatabaseUploader(
            database.GetCollection<BsonDocument>("Logs"),
            workspace.Injections.RequireInjection<ISerializationProvider>())
        {
            MinimumLevel = LogLevel.Information
        });
        logCenter.LogHandlers.Add(typeof(ConsolePrinter), new ConsolePrinter(
            workspace.Injections.RequireInjection<ISerializationProvider>())
        {
            MinimumLevel = LogLevel.Warning
        });

        var models = new List<ModelConfiguration>();
        models.AddOpenAiModels();
        models.AddOllamaModels();
        
        var selectedModelName = AnsiConsole.Prompt(
            new SelectionPrompt<string>()
                .Title("Select [blue]Model[/]:")
                .PageSize(10)
                .MoreChoicesText("[grey](See More)[/]")
                .AddChoices(models.Select(model => model.Name)));
        var selectedConfiguration = models.First(model => model.Name == selectedModelName);

        var demos = new Dictionary<string, Type>
        {
            { "Titanic", typeof(DemoTitanic) },
            { "Mushroom", typeof(DemoMushroom) },
            { "Horses", typeof(DemoHorses) },
            { "Insurance", typeof(DemoInsurance) },
            { "CarPrices", typeof(DemoCarPrices) },
            { "MohsHardness", typeof(DemoMohs) },
            { "TitanicRAG-1", typeof(DemoTitanicRag1)},
            { "TitanicRAG-2", typeof(DemoTitanicRag2)}
        };
        
        var selectedDemoName = AnsiConsole.Prompt(
            new SelectionPrompt<string>()
                .Title("Select [blue]Demo[/]:")
                .PageSize(10)
                .MoreChoicesText("[grey](See More)[/]")
                .AddChoices(demos.Keys));
        
        
        var trainingCount = AnsiConsole.Ask<int>("Input [red]Training Count[/]:");
        var enableSubstitutionScript = AnsiConsole.Prompt(
            new ConfirmationPrompt("Enable [blue]Substitution Script[/]?")
            {
                ShowChoices = true,
                DefaultValue = false,
                ShowDefaultValue = true
            });
        var compressionThreshold = AnsiConsole.Ask<int?>(
            "Input [blue]Compression Threshold[/]:", null);
        var replacementThreshold = AnsiConsole.Ask<int?>(
            "Input [blue]Compression Threshold[/]:", null);
        
        await workspace.Startup();

        workspace.Injections.AddOpenSourceAgent(
            new Uri(selectedConfiguration.ApiUrl),
            new ApiKeyCredential(selectedConfiguration.ApiKey),
            selectedConfiguration.Name
        );
        
        var demo = (IDemo)Activator.CreateInstance(demos[selectedDemoName])!;

        await demo.Run(workspace, new TestConfiguration()
        {
            ModelName = selectedConfiguration.Name,
            TrainingCount = trainingCount,
            EnableSubstitutionScript = enableSubstitutionScript,
            EnableStructuredOutput = selectedConfiguration.SupportStructuredOutput,
            CompressionThreshold = compressionThreshold,
            ReplacementThreshold = replacementThreshold
        });
        
        await workspace.Shutdown();
    }
}