﻿using System.Collections;
using System.Collections.Concurrent;
using OpenAI.Chat;

namespace Robotless.Modules.Agenting;

public class AgentMemory : IAgentMemory
{
    /// <summary>
    /// Main branch of the memory that this memory belongs to.
    /// </summary>
    public AgentMemory? Trunk { get; private set; }

    private readonly List<ChatMessage> _messages = [];
    
    private IEnumerable<ChatMessage>? _enumerable;

    /// <summary>
    /// Ad a message into this memory.
    /// </summary>
    public virtual void Add(ChatMessage message)
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        _messages.Add(message);
    }

    /// <summary>
    /// Remove a message from this memory.
    /// </summary>
    public virtual void Remove(ChatMessage message)
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        _messages.Remove(message);
    }

    public virtual void Clear()
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        _messages.Clear();
    }

    /// <summary>
    /// Commit all messages in this memory to the trunk memory.
    /// After committing, this memory will be cleared.
    /// </summary>
    public virtual void Commit()
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        if (Trunk == null)
            throw new InvalidOperationException("This memory is not a branch, does not have a trunk to commit to.");
        Trunk._messages.AddRange(_messages);
        _messages.Clear();
    }
    
    /// <summary>
    /// Create a new scope memory from this memory, which will inherit messages from this memory.
    /// </summary>
    public virtual AgentMemory Scope()
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        var scope = Allocate();
        scope.Trunk = this;
        scope._enumerable = _messages.Concat(scope._messages);
        return scope;
    }

    private static readonly ConcurrentBag<AgentMemory> MemoryPool = [];
    
    /// <summary>
    /// The max count of reusable memory stored in the memory pool.
    /// </summary>
    public static int MaxStoredCount { get; set; } = Environment.ProcessorCount * 2;
    
    protected bool Recycled;
    
    /// <summary>
    /// Allocate a new memory from the memory pool.
    /// </summary>
    public static AgentMemory Allocate()
    {
        var memory = MemoryPool.TryTake(out var instance) ? instance : new AgentMemory();
        memory.Recycled = false;
        memory.Reset();
        return memory;
    }
    
    /// <summary>
    /// Clear all messages stored in this memory, reset other status,
    /// and return it to the memory pool.
    /// </summary>
    public virtual void Dispose()
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        if (MemoryPool.Count >= MaxStoredCount)
            return;
        Recycled = true;
        Reset();
        MemoryPool.Add(this);
    }

    protected void Reset()
    {
        _messages.Clear();
        _enumerable = null;
        Trunk = null;
    }
    
    IEnumerator IEnumerable.GetEnumerator()
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        return _enumerable?.GetEnumerator() ?? _messages.GetEnumerator();
    }

    public virtual IEnumerator<ChatMessage> GetEnumerator()
    {
        if (Recycled)
            throw new ObjectDisposedException(nameof(AgentMemory));
        return _enumerable?.GetEnumerator() ?? _messages.GetEnumerator();
    }
    
    /// <summary>
    /// Messages from the trunk memory will not be included in this list.
    /// </summary>
    public IReadOnlyList<ChatMessage> LocalMessages => _messages;
}