﻿using System.Numerics;

namespace Robotless.Modules.Mocking.Learning.Metrics;

public class ErrorRatioMetrics<TResult>(
    bool enableGreen = true,
    double yellowThreshold = 0.5,
    double redThreshold = 1.0) : TrainerMetrics<TResult>
    where TResult : INumber<TResult>
{
    public double ErrorRatio { get; private set; }
    
    public override void Report(TResult expectedResult, TResult actualResult)
    {
        var difference = Math.Abs(Convert.ToDouble(expectedResult - actualResult));
        ErrorRatio = difference / Convert.ToDouble(expectedResult);
    }

    public override string ToText()
        => ErrorRatio.ToString("P2");

    public override string ToMarkup()
    {
        var text = ToText();
        if (ErrorRatio > redThreshold)
            return $"[red]{text}[/]";
        if (ErrorRatio > yellowThreshold)
            return $"[yellow]{text}[/]";
        return enableGreen ? $"[green]{text}[/]" : text;
    }
}

public static class ErrorRatioMetricsExtensions
{
    public static Dictionary<string,Func<TrainerMetrics<TResult>>> UseErrorRatio<TResult>(
        this Dictionary<string,Func<TrainerMetrics<TResult>>> metrics, 
        bool enableGreen = true, double yellowThreshold = 0.5, double redThreshold = 1.0)
        where TResult : INumber<TResult>
    {
        metrics["ErrorRatio"] = () => new ErrorRatioMetrics<TResult>(enableGreen, yellowThreshold, redThreshold);
        return metrics;
    }
}