using System.Runtime.InteropServices;
using JetBrains.Annotations;

namespace Robotless.Modules.Mocking.Learning;

public class TrainerSession<TResult>(
    IReadOnlyDictionary<string, TrainerMetrics<TResult>> metrics,
    Stage stage,
    int? count = null)
{
    public Stage Stage { get; } = stage;

    public IReadOnlyDictionary<string, TrainerMetrics<TResult>> Metrics { get; } = metrics;
    
    /// <summary>
    /// Count of results reported.
    /// For metrics, this index starts from 1.
    /// </summary>
    public int Index { get; private set; }

    /// <summary>
    /// Total count of data entries to run in this session.
    /// </summary>
    public int? Count { get; private set; } = count;

    public void ReportResult(TResult expectedResult, TResult actualResult)
    {
        ++Index;
        foreach (var metric in Metrics.Values)
            metric.Report(expectedResult, actualResult);
    }

    private TimeSpan _durationSummary = TimeSpan.Zero;
    
    private int _durationCount = 0;

    public TimeSpan AverageDuration => _durationCount != 0 ? _durationSummary / _durationCount : TimeSpan.Zero;
    
    public void ReportDuration(TimeSpan duration)
    {
        _durationCount++;
        _durationSummary += duration;
    }
    
    public IReadOnlyDictionary<string, string> ToStringMetrics()
        => Metrics.Select(
            pair => new KeyValuePair<string, string>(
                pair.Key, pair.Value.ToText())).ToDictionary();
}