using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.AiAgent;
using Robotless.Modules.Documenting;
using Robotless.Modules.Serializing;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Mocking;

public abstract class MockDelegate : MockFunction
{
    private readonly (string Name, ISnapshotSerializer Serializer)[] _argumentSerializers;
    
    private readonly ISnapshotSerializer _resultSerializer;

    public MockDelegate(Type delegateType, IAgent executor, IAgent? reflector, IAgent? generator, 
        ISerializationProvider serialization, IDocumentation? documentation = null) : 
        base(executor, reflector, generator)
    {
        var typeDocumentation = documentation?.GetEntry(delegateType);
        var command = string.Empty;
        if (typeDocumentation?.Summary != null)
            command = typeDocumentation.Summary + "\n";
        if (typeDocumentation?.Remarks != null)
            command += "Remarks: " + typeDocumentation.Remarks + "\n";

        var method = delegateType.GetMethod("Invoke")!;
        var parameters = method.GetParameters();
        var argumentsProperties = new BsonDocument();
        var argumentsSchema = new BsonDocument()
        {
            { "type", "object" },
            { "properties", argumentsProperties },
            { "additionalProperties", false },
            { "required", new BsonArray(parameters.Select(parameter => parameter.Name)) }
        };
        _argumentSerializers = new (string Name, ISnapshotSerializer Serializer)[method.GetParameters().Length];
        
        for (var parameterIndex = 0; parameterIndex < parameters.Length; ++parameterIndex)
        {
            var parameter = parameters[parameterIndex];
            var serializer = serialization.RequireSerializer(parameter.ParameterType);
            _argumentSerializers[parameterIndex] = (parameter.Name!, serializer);
            var parameterSchema = new BsonDocument();
            var writer = new SchemaWriter(new BsonDocumentWriter(parameterSchema));
            writer.WriteStartDocument();
            serializer.GenerateJsonSchema(writer);
            writer.WriteEndDocument();
            argumentsProperties[parameter.Name!] = parameterSchema;
        }
        
        foreach (var parameterEntry in typeDocumentation?.Parameters ?? Enumerable.Empty<IParameterEntry>())
            argumentsProperties[parameterEntry.ParameterName]["description"] = parameterEntry.Description;
        
        _resultSerializer = serialization.RequireSerializer(method.ReturnType);
        var resultSchema = new BsonDocument();
        var resultSchemaWriter = new SchemaWriter(new BsonDocumentWriter(resultSchema));
        resultSchemaWriter.WriteStartDocument();
        _resultSerializer.GenerateJsonSchema(resultSchemaWriter);
        resultSchemaWriter.WriteEndDocument();

        UpdatePrompt(command, argumentsSchema, resultSchema);
    }
    
    public BsonDocument SerializeArguments(object?[] arguments)
    {
        var argumentsDocument = new BsonDocument();
        var argumentsWriter = new BsonDocumentWriter(argumentsDocument);
        argumentsWriter.WriteStartDocument();
        for (var parameterIndex = 0; parameterIndex < arguments.Length; ++parameterIndex)
        {
            var (name, serializer) = _argumentSerializers[parameterIndex];
            argumentsWriter.WriteName(name);
            var argument = arguments[parameterIndex];
            if (argument == null)
                argumentsWriter.WriteNull();
            else
                serializer.SaveSnapshot(argument, argumentsWriter);
        }
        argumentsWriter.WriteEndDocument();
        return argumentsDocument;
    }

    public BsonValue SerializeResults(object result)
    {
        var writer = new BsonValueWriter();
        _resultSerializer.SaveSnapshot(result, writer);
        return writer.Value;
    }
    
    public object? Invoke(object?[] arguments)
    {
        var invocation = MakeInvocation(arguments);
        return DeserializeResults(invocation.Results);
    }
    
    public MockInvocation MakeInvocation(object?[] arguments)
    {
        // Serializer arguments.
        var argumentsDocument = SerializeArguments(arguments);
        // Get response.
        var invocation = Task.Run(() => base.MakeInvocation(argumentsDocument)).Result;
        return invocation;
    }
    
    public object? DeserializeResults(BsonValue results)
    {
        _resultSerializer.NewInstance(out var result);
        _resultSerializer.LoadSnapshot(ref result, new SnapshotReader(new BsonValueReader(results)));
        return result;
    }

    public void Example(object?[] arguments, object result, string remarks)
    {
        var bsonArguments = SerializeArguments(arguments);

        var bsonResults = new BsonValueWriter();
        _resultSerializer.SaveSnapshot(result, bsonResults);
        base.Example(bsonArguments, bsonResults.Value, remarks);
    }
    
    public void Amend(object?[] arguments, object result, string? remarks)
    {
        var bsonArguments = SerializeArguments(arguments);

        var bsonResults = new BsonValueWriter();
        _resultSerializer.SaveSnapshot(result, bsonResults);
        base.Amend(bsonArguments, bsonResults.Value, remarks);
    }
    
    public Task<string> Reflect(object?[] arguments, object wrongResult, string wrongRemarks, object correctResult)
    {
        return base.Reflect(SerializeArguments(arguments), 
            SerializeResults(wrongResult), 
            wrongRemarks,
            SerializeResults(correctResult)
        );
    }
}