using JetBrains.Annotations;
using MongoDB.Bson;
using Robotless.Modules.OpenAi.Chat;
using Robotless.Framework;
using Robotless.Modules.AiAgent;
using Robotless.Modules.AiAgent.Messages;
using Robotless.Modules.Logging;
using Robotless.Modules.Serializing;

namespace Robotless.Modules.Mocking;

public class MockFunction : Entity
{
    /// <summary>
    /// Serialization context for this mock function.
    /// </summary>
    public required ISerializationProvider Serialization { get; init; }

    /// <summary>
    /// Agent for execution.
    /// </summary>
    public IAgent ExecutionAgent { get; }

    /// <summary>
    /// Agent for reflection.
    /// </summary>
    public IAgent? ReflectionAgent
    {
        get => field ?? ExecutionAgent;
        set;
    }

    /// <summary>
    /// Agent for script generation.
    /// </summary>
    public IAgent? GenerationAgent
    {
        get => field ?? ExecutionAgent;
        set
        {
            field = value;
            Script.Agent = value ?? ExecutionAgent;
        }
    }

    /// <summary>
    /// Logger for this mock function.
    /// </summary>
    [ComponentDependency]
    public LoggerComponent? Logger { get; init; }

    /// <summary>
    /// Memory of this mock function.
    /// </summary>
    public MockMemory Memory { get; set; } = [];

    /// <summary>
    /// Tools for this function to use.
    /// </summary>
    public Dictionary<string, IAgentTool> Tools { get; } = new();

    /// <summary>
    /// Options to control the completion behavior.
    /// </summary>
    public readonly ChatCompletionOptions Options = new();

    public MockScript Script { get; }

    /// <summary>
    /// If true, then this mock function will try to use the substitution script to accelerate the invocation.
    /// It is false by default.
    /// </summary>
    public bool EnableSubstitutionScript
    {
        get;
        set
        {
            if (field == value)
                return;
            field = value;
            UpdatePrompt(_signatureCommand, _signatureArgumentsSchema, _signatureResultsSchema);
        }
    }

    public bool EnableStructuredOutput
    {
        get;
        set
        {
            if (field == value)
                return;
            field = value;
            // Options.ResponseFormat = value ? _signatureResponseFormat : ChatResponseFormat.CreateJsonObjectFormat();
            Options.ResponseFormat = value ? _signatureResponseFormat : ChatResponseFormat.CreateTextFormat();
        }
    }
    
    public MockFunction(IAgent executor, IAgent? reflector, IAgent? generator)
    {
        ExecutionAgent = executor;
        ReflectionAgent = reflector ?? executor;
        GenerationAgent = generator ?? executor;
        Script = new MockScript(generator ?? executor, Memory.NewBranch());
    }
    
    private string _signatureCommand = default!;
    private BsonDocument _signatureArgumentsSchema = default!;
    private BsonDocument _signatureResultsSchema = default!;
    private ChatResponseFormat _signatureResponseFormat = default!;

    /// <summary>
    /// Invoke this method to change the command, arguments schema and results schema of this mock function.
    /// </summary>
    /// <param name="command">New command for this mock function.</param>
    /// <param name="argumentsSchema">New arguments schema for this mock function.</param>
    /// <param name="resultsSchema">New results schema for this mock function. </param>
    [PublicAPI]
    public void UpdatePrompt(string command, BsonDocument argumentsSchema, BsonDocument resultsSchema)
    {
        _signatureCommand = command;
        _signatureArgumentsSchema = argumentsSchema;
        _signatureResultsSchema = resultsSchema;
        var (prompt, requestSchema, responseSchema, responseFormat) =
            MockInvocation.BuildInvocationPrompt(command, argumentsSchema, resultsSchema, EnableSubstitutionScript);
        Memory.Prompt.Text = prompt;
        _signatureResponseFormat = responseFormat;
        if (EnableStructuredOutput)
            Options.ResponseFormat = responseFormat;
        Script.Update(requestSchema, responseSchema);
    }

    /// <summary>
    /// Invoke this mock function.
    /// </summary>
    /// <param name="arguments">Arguments of this invocation.</param>
    /// <param name="cancellation">Token to cancel the invocation.</param>
    /// <returns>The return value of this invocation.</returns>
    [PublicAPI]
    public async Task<BsonValue> Invoke(BsonDocument arguments, CancellationToken cancellation = default)
    {
        var invocation = await MakeInvocation(arguments, cancellation);
        return invocation.Results;
    }

    [PublicAPI]
    public async Task<MockInvocation> MakeInvocation(BsonDocument arguments,
        CancellationToken cancellation = default)
    {
        MockInvocation? invocation;

        var requestMessage = new AgentRequestMessage(arguments.ToString());

        if (EnableSubstitutionScript && Script.Functor != null)
        {
            try
            {
                var response = await Script.Functor(arguments);
                invocation = new MockInvocation(arguments, response);
                Memory.Add(invocation.RequestMessage);
                Memory.Add(invocation.ResponseMessage);
                Memory.Add(invocation);
                return invocation;
            }
            catch (Exception error)
            {
                Logger?.LogException(error, "Failed to execute the script.");
                Script.Memory.AddUserMessage("Exception occurs while executing the script: " + error.Message);
            }
        }

        var responseMessage = await ExecutionAgent.Chat(requestMessage, Memory, Options, Tools, cancellation);
        invocation = new MockInvocation(requestMessage, responseMessage);
        Memory.Add(invocation);
        Memory.ReportTokenUsage(
            invocation.ResponseMessage.TokenUsage?.InputTokenCount, "Invocation.Input");
        Memory.ReportTokenUsage(
            invocation.ResponseMessage.TokenUsage?.OutputTokenCount, "Invocation.Output");
        
        if (EnableSubstitutionScript && invocation.IsReadyToCompile)
        {
            await Script.Build(cancellation);
        }

        return invocation;
    }

    /// <summary>
    /// Amend invocations with the specified arguments.
    /// </summary>
    /// <remarks>This will not affect invocations in the trunk memory.</remarks>
    /// <param name="arguments">Arguments to match the invocations.</param>
    /// <param name="results">If it is not null, results will be set to it.</param>
    /// <param name="remarks">If it is not null, remarks will be set to it.</param>
    /// <returns>Count of invocations that have been amended.</returns>
    [PublicAPI]
    public int Amend(BsonDocument arguments, BsonValue? results, BsonValue? remarks)
    {
        var count = 0;
        foreach (var invocation in Memory.Invocations)
        {
            if (invocation.Arguments != arguments)
                continue;
            ++count;
            if (remarks != null)
                invocation.Remarks = remarks;
            if (results != null)
                invocation.Results = results;
            invocation.UpdateText();
        }

        if (count == 0)
            return count;
        Script.Clear();
        return count;
    }

    /// <summary>
    /// Add an example into current mock memory.
    /// </summary>
    /// <param name="arguments">Arguments of this example invocation.</param>
    /// <param name="results">Results of this example invocation.</param>
    /// <param name="remarks">Remarks for this example invocation.</param>
    [PublicAPI]
    public void Example(BsonDocument arguments, BsonValue results, BsonValue remarks)
    {
        var invocation = new MockInvocation(arguments, results, remarks);
        Memory.Add(invocation.RequestMessage);
        Memory.Add(invocation.ResponseMessage);
        Memory.Add(invocation);
    }

    /// <summary>
    /// Perform a reflection in current memory scope.
    /// </summary>
    /// <returns>Notes summarized by agent on the provided invocation.</returns>
    [PublicAPI]
    public async Task<string> Reflect(BsonDocument arguments, BsonValue wrongResults, BsonValue wrongRemarks,
        BsonValue correctResults, CancellationToken cancellation = default)
    {
        var request = new AgentRequestMessage(
            $"""
             In previous invocation when the arguments are:
             {arguments.ToJson()}
             You gave the wrong results:
             {wrongResults.ToJson()}
             At that time, the reasoning you wrote for this wrong results is:
             {wrongRemarks.ToJson()}
             But the correct results should be:
             {correctResults.ToJson()}
             What do you think led you to make these mistakes? 
             What notes can you write down to help your future self avoid making these mistakes?
             Output them in plain text.
             """);
        var response = await ReflectionAgent.Chat(request, Memory, cancellation: cancellation);
        Memory.ReportTokenUsage(response.TokenUsage?.InputTokenCount, "Reflection.Input");
        Memory.ReportTokenUsage(response.TokenUsage?.OutputTokenCount, "Reflection.Output");
        return response.Text;
    }
}

public static class MockFunctionExtensions
{
    public readonly struct MemorySwapScope : IDisposable
    {
        private readonly MockFunction _function;
        private readonly MockMemory _memory;

        public MockMemory ScopeMemory { get; }

        public MemorySwapScope(MockFunction function)
        {
            _function = function;
            _memory = function.Memory;
            ScopeMemory = _memory.NewBranch();
            function.Memory = ScopeMemory;
        }

        public void Dispose()
        {
            _function.Memory = _memory;
        }
    }

    public static MemorySwapScope NewMemoryScope(this MockFunction function, out MockMemory scope)
    {
        var swapScope = new MemorySwapScope(function);
        scope = swapScope.ScopeMemory;
        return swapScope;
    }
}