using System.Text;
using MongoDB.Bson;
using Robotless.Modules.OpenAi.Chat;
using Robotless.Modules.AiAgent.Messages;
using Robotless.Modules.Mocking.Utilities;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Mocking;

public class MockInvocation
{
    public AgentRequestMessage RequestMessage { get; }
    
    public AgentResponseMessage ResponseMessage { get; }

    private BsonDocument _requestDocument;

    private BsonDocument _responseDocument;
    
    public BsonDocument Arguments
    {
        get => _requestDocument;
        set => _requestDocument = value;
    }
    
    public BsonValue Results
    {
        get => _responseDocument[FieldNameResults];
        set => _responseDocument[FieldNameResults] = value;
    }

    public BsonValue Remarks
    {
        get => _responseDocument[FieldNameRemarks];
        set => _responseDocument[FieldNameRemarks] = value;
    }
    
    public bool IsReadyToCompile
    {
        get => _responseDocument.TryGetValue(FieldNameCompile, out var ready)
               && ready.AsBoolean;
        set => _responseDocument[FieldNameCompile] = value ? BsonBoolean.True : BsonBoolean.False;
    }

    /// <summary>
    /// The token usage of this invocation.
    /// It will be 0 if the token usage is not available.
    /// </summary>
    public int TokenUsage => ResponseMessage.TokenUsage?.TotalTokenCount ?? 0;
    
    public MockInvocation(AgentRequestMessage requestMessage, AgentResponseMessage responseMessage)
    {
        RequestMessage = requestMessage;
        ResponseMessage = responseMessage;
        
        _requestDocument = BsonDocument.Parse(requestMessage.Text);
        _responseDocument = BsonDocument.Parse(responseMessage.Text);
    }
    
    public MockInvocation(BsonDocument requestDocument, BsonDocument responseDocument)
    {
        RequestMessage = new AgentRequestMessage(requestDocument.ToJson());
        ResponseMessage = new AgentResponseMessage(responseDocument.ToJson());
        
        _requestDocument = requestDocument.DeepClone().AsBsonDocument;
        _responseDocument = responseDocument.DeepClone().AsBsonDocument;
    }
    
    public MockInvocation(BsonDocument arguments, BsonValue results, BsonValue remarks)
    {
        _requestDocument = arguments.DeepClone().AsBsonDocument;
        _responseDocument = new BsonDocument
        {
            [FieldNameRemarks] = remarks,
            [FieldNameResults] = results,
        };
        RequestMessage = new AgentRequestMessage(_requestDocument.ToJson());
        ResponseMessage = new AgentResponseMessage(_responseDocument.ToJson());
    }
    
    /// <summary>
    /// The message text cannot automatically update when the value is changed,
    /// therefore users have to invoke this method to update the message text manually.
    /// </summary>
    public void UpdateText()
    {
        RequestMessage.Text = _requestDocument.ToJson();
        ResponseMessage.Text = _responseDocument.ToJson();
    }
    
    /// <summary>
    /// Additional notes for this invocation.
    /// </summary>
    public IDictionary<string, object> Notes { get; } = new Dictionary<string, object>();

    public object? this[string key]
    {
        get => Notes.TryGetValue(key, out var value) ? value : null;
        set
        {
            if (value != null)
                Notes[key] = value;
            else
                Notes.Remove(key);
        }
    }
    
    public static (string Prompt, string RequestSchema, string ResponseSchema, ChatResponseFormat ResponseFormat) 
        BuildInvocationPrompt(string command, BsonDocument argumentsSchema, BsonDocument resultsSchema, 
            bool enableScript = false)
    {
        var responseSchema = new ObjectSchemaDocument()
            .WithProperty(MockInvocation.FieldNameRemarks, SchemaType.String,
                "Comments about the results, such as why the results are the way they are.")
            .WithProperty(MockInvocation.FieldNameResults, resultsSchema);
        if (enableScript)
            responseSchema.WithProperty(MockInvocation.FieldNameCompile, SchemaType.Boolean,
                "Set this option to true if you have got enough information" +
                "to write a C# script for this function.");
        
        var requestSchemaText = argumentsSchema.ToJson();
        var responseSchemaText = responseSchema.ToJson();
        
        var builder = new StringBuilder();
        builder.AppendLine(
            "You are a mock function. " +
            "You can output the return value based on the arguments according to the user designated function. ");
        builder.AppendLine("The function you are mocking is:");
        builder.AppendLine(command);
        builder.AppendLine("The JSON Schema of the input is:");
        builder.AppendLine(argumentsSchema.ToJson());
        builder.AppendLine("The JSON Schema of the output is:");
        builder.AppendLine(responseSchema.ToJson());
        builder.AppendLine("Only output the JSON content matching the response schema.");
        var prompt = builder.ToString();
        
        return (prompt, requestSchemaText, responseSchemaText, 
            responseSchema.ToResponseFormat("Invocation"));
    }
    
    private const string FieldNameRemarks = "Remarks";
    private const string FieldNameResults = "Results";
    private const string FieldNameCompile = "IsReadyToCompile";
}