using System.Collections;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Runtime.InteropServices;
using JetBrains.Annotations;
using Robotless.Modules.OpenAi.Chat;
using Robotless.Modules.AiAgent;
using Robotless.Modules.AiAgent.Messages;

namespace Robotless.Modules.Mocking;

[DebuggerDisplay("Count = {Count}")]
public class MockMemory : IAgentMemory
{
    /// <summary>
    /// Main branch of the memory that this memory belongs to.
    /// </summary>
    [PublicAPI] public MockMemory? Trunk { get; }

    public AgentSystemMessage Prompt { get; }
    
    public MockMemory(MockMemory? trunk = null)
    {
        Trunk = trunk;
        
        if (trunk != null)
        {
            _enumerable = trunk.Concat(_messages);
            _usages = trunk._usages;
            Prompt = trunk.Prompt;
        }
        else
        {
            _usages = new Dictionary<string, int>();
            Prompt = new AgentSystemMessage("");
            _messages.AddFirst(Prompt);
        }
    }

    /// <summary>
    /// The total count of messages in this memory and all its branches.
    /// </summary>
    public int Count => (Trunk?.Count ?? 0) + _messages.Count;

    /// <summary>
    /// Collection of invocations in this memory.
    /// </summary>
    public IReadOnlyCollection<MockInvocation> Invocations => _invocations;
    
    /// <summary>
    /// Ad a message into this memory.
    /// </summary>
    public virtual void Add(ChatMessage message)
    {
        _messages.AddLast(message);
    }

    /// <summary>
    /// Remove a message from this memory.
    /// </summary>
    public virtual void Remove(ChatMessage message)
    {
        _messages.Remove(message);
    }

    /// <summary>
    /// Clear the messages and invocations.
    /// </summary>
    public virtual void Clear()
    {
        _messages.Clear();
        _invocations.Clear();
        _messages.AddFirst(Prompt);
    }

    public virtual void ClearInvocations()
    {
        foreach (var invocation in _invocations)
        {
            _messages.Remove(invocation.RequestMessage);
            _messages.Remove(invocation.ResponseMessage);
        }
        _invocations.Clear();
    }
    
    /// <summary>
    /// Add a mock invocation to this memory.
    /// </summary>
    /// <remarks>The request and response messages will not be added into this memory.</remarks>
    /// <param name="invocation">Mock invocation to add.</param>
    public void Add(MockInvocation invocation)
    {
        /* Note:
         * IAgent will automatically add the request and the response into the chat history,
         * so we don't need to add them into the memory here.
         */
        _invocations.AddLast(invocation);
    }

    /// <summary>
    /// Remove a mock invocation and its request and response messages from this memory.
    /// </summary>
    /// <param name="invocation">Mock invocation to remove.</param>
    public void Remove(MockInvocation invocation)
    {
        _invocations.Remove(invocation);
        _messages.Remove(invocation.RequestMessage);
        _messages.Remove(invocation.ResponseMessage);
    }
    
    /// <summary>
    /// Commit all messages in this memory to the trunk memory.
    /// After committing, this memory will be cleared.
    /// </summary>
    public virtual void Commit()
    {
        if (Trunk == null)
            throw new InvalidOperationException("This memory is not a branch, it has no trunk to commit to.");
        // Merge messages.
        foreach (var message in _messages)
        {
            Trunk._messages.AddLast(message);
        }
        _messages.Clear();
        // Merge invocations.
        foreach (var invocation in _invocations)
        {
            Trunk._invocations.AddLast(invocation);
        }
        _invocations.Clear();
    }
    
    /// <summary>
    /// Create a new scope memory from this memory, which will inherit messages from this memory.
    /// </summary>
    public virtual MockMemory NewBranch()
    {
        return new MockMemory(this);
    }

    public IReadOnlyDictionary<string, int> TokenUsages => _usages;

    /// <summary>
    /// Report a token usage.
    /// </summary>
    /// <param name="count">Count of tokens.</param>
    /// <param name="category">Category of this usage, default category is "Uncategorized".</param>
    public void ReportTokenUsage([ValueRange(1, int.MaxValue)] int? count, string? category = null)
    {
        if (count is 0 or null)
            return;
        category ??= "Uncategorized";
        CollectionsMarshal.GetValueRefOrAddDefault(_usages, category, out _) += count.Value;
    }
    
    IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();

    public IEnumerator<ChatMessage> GetEnumerator() => (_enumerable ?? _messages).GetEnumerator();

    private readonly LinkedList<ChatMessage> _messages = [];

    private readonly LinkedList<MockInvocation> _invocations = [];
    
    private readonly IEnumerable<ChatMessage>? _enumerable;

    private readonly Dictionary<string, int> _usages;
    
    public IReadOnlyCollection<ChatMessage> LocalMessages => _messages;
}