﻿using System.Diagnostics.CodeAnalysis;
using System.Text;
using System.Text.RegularExpressions;
using JetBrains.Annotations;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Scripting;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Scripting;
using Microsoft.Extensions.Logging;
using MongoDB.Bson;
using Robotless.Modules.OpenAi.Chat;
using Robotless.Modules.AiAgent;
using Robotless.Modules.AiAgent.Messages;
using Robotless.Modules.Injecting;
using Robotless.Modules.Logging;
using Robotless.Modules.Mocking.Utilities;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Mocking;

public delegate Task<BsonDocument> ScriptFunctor(BsonDocument arguments);

public class MockScript(IAgent agent, MockMemory memory)
{
    public ChatCompletionOptions Options { get; } = new()
    {
        ResponseFormat = new ObjectSchemaDocument()
            .WithType(SchemaType.Object)
            .WithProperty("Script", SchemaType.String, "The generated C# script.")
            .ToResponseFormat("ScriptResponse")
    };
    
    [Injection] public LoggerComponent? Logger { get; init; }
    
    public int MaxRetries { get; set; } = 3;

    public MockMemory Memory { get; } = memory;
    
    public string? Code { get; private set; }
    
    public ScriptFunctor? Functor { get; private set; }

    public IAgent Agent { get; set; } = agent;

    [PublicAPI]
    public Action<string?>? OnCodeChanged;

    private string _inputSchema = null!;
    private string _outputSchema = null!;
    
    public void Update(string inputSchema, string outputSchema)
    {
        _inputSchema = inputSchema;
        _outputSchema = outputSchema;
        Clear();
    }

    public void Clear()
    {
        var changed = Code != null;
        Code = null;
        Functor = null;
        if (changed)
            OnCodeChanged?.Invoke(null);
    }
    
    public async Task<bool> Build(CancellationToken cancellation = default)
    {
        // The script generation process is also recorded in the chat history as an activity.
        var request = new AgentRequestMessage(
            $"""
             Based on the command and history of input-output, please provide a C# script to implement the function.
             Generate a function with the following signature:
             ```csharp
             public static BsonDocument MockFunction(BsonDocument Arguments)
             ```
             The JSON schema of the `Arguments` parameter is as follows:
             {_inputSchema}
             The JSON schema of the return value should be as follows:
             {_outputSchema}
             """);
        for (var retry = 0; retry < MaxRetries; ++retry)
        {
            var response = await Agent.Chat(request, Memory, Options, null, cancellation);
            
            Memory.ReportTokenUsage(response.TokenUsage?.InputTokenCount ?? 0, "ScriptGeneration.Input");
            Memory.ReportTokenUsage(response.TokenUsage?.OutputTokenCount ?? 0, "ScriptGeneration.Output");

            var bsonResponse = BsonDocument.Parse(response.Text);
            // Extract the script.
            var scriptCode = ParseScript(bsonResponse["Script"].AsString, "MockFunction");
            if (scriptCode == null)
            {
                Logger?.LogError("Failed to find the static method `MockFunction` in the script:\n{Script}",
                    response.Text ?? "[No Content]");
                Memory.AddUserMessage("Error: cannot find static method `MockFunction` in the script.");
                continue;
            }
            response.Text = scriptCode;
            
            var codeBuilder = new StringBuilder();
            codeBuilder.AppendLine(scriptCode);
            codeBuilder.AppendLine("return MockFunction(Arguments);");
            scriptCode = codeBuilder.ToString();
            
            // Compile the script.
            try
            {
                var script = CSharpScript
                    .Create<BsonDocument>(scriptCode, globalsType: typeof(ScriptArguments), options:
                        ScriptOptions.Default
                            .WithReferences(typeof(BsonDocument).Assembly)
                            .WithImports(typeof(BsonDocument).Namespace));
                script.Compile();
                Logger?.LogTrace("The script has been compiled with following script:\n {Script}", scriptCode);
                Code = scriptCode;
                Functor = ScriptArguments.Wrap(script.CreateDelegate());
                OnCodeChanged?.Invoke(Code);
                return true;
            }
            catch (CompilationErrorException exception)
            {
                Logger?.LogException(exception, "Failed to compile the script.", 
                    details: details => details.Script = scriptCode);
                Memory.AddUserMessage("Error: failed to compile the script:\n" + exception.Message);
            }
        }

        return false;
    }
    
    private static bool TrySearchMethodCode(SyntaxNode root, string identifier,
        [NotNullWhen(true)] out string? code)
    {
        switch (root)
        {
            case MethodDeclarationSyntax method when method.Identifier.Text == identifier:
                code = method.ToFullString();
                return true;
            case LocalFunctionStatementSyntax function when function.Identifier.Text == identifier:
                code = function.ToFullString();
                return true;
            case CompilationUnitSyntax or GlobalStatementSyntax or TypeDeclarationSyntax:
                foreach (var node in root.ChildNodes())
                {
                    if (TrySearchMethodCode(node, identifier, out code))
                        return true;
                }
                break;
        }
        code = null;
        return false;
    }
    
    public static string? ParseScript(string code, string methodName)
    {
        code = Regex.Unescape(Regex.Unescape(code));
        var codeBuilder = new StringBuilder();
        var syntaxTree = CSharpSyntaxTree.ParseText(code);
        var unitRoot = syntaxTree.GetCompilationUnitRoot();

        foreach (var usingDirective in unitRoot.Usings)
            codeBuilder.AppendLine(usingDirective.ToString());

        if (!TrySearchMethodCode(syntaxTree.GetCompilationUnitRoot(),
                methodName, out var methodCode))
            return null;

        codeBuilder.AppendLine(methodCode);
        
        return codeBuilder.ToString();
    }
    
    public class ScriptArguments(BsonDocument arguments)
    {
        public BsonDocument Arguments = arguments;
    
        public static ScriptFunctor Wrap(ScriptRunner<BsonDocument> runner)
            => arguments => runner.Invoke(new ScriptArguments(arguments));
    }
}