using System.Reflection;
using System.Reflection.Emit;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Mocking.Utilities;

public class InvokerRedirectorGenerator
{
    private static readonly DynamicTypeCache RedirectorTypes = new(GenerateRedirector);

    private static Type GenerateRedirector(ModuleBuilder module, Type delegateType)
    {
        var typeBuilder = module.DefineType("MockDelegate_" + delegateType.Name,
                TypeAttributes.Public | TypeAttributes.AutoClass | TypeAttributes.BeforeFieldInit);
        
        var functionField = typeBuilder.DefineField("MockFunction", typeof(MockDelegate), FieldAttributes.Public);
        GenerateConstructor(typeBuilder, functionField);
        GenerateRedirector(typeBuilder, functionField, delegateType);
        return typeBuilder.CreateType();
    }
    
    // Generate a constructor which stores a MockDelegate instance into a field.
    private static void GenerateConstructor(TypeBuilder typeBuilder, FieldBuilder functionField)
    {
        var constructorBuilder = typeBuilder.DefineConstructor(
            MethodAttributes.HideBySig | MethodAttributes.SpecialName | 
            MethodAttributes.RTSpecialName | MethodAttributes.Public, CallingConventions.Standard,
            [typeof(MockDelegate)]);
        var code = constructorBuilder.GetILGenerator();
        
        // Store the mock function object into the field.
        code.Emit(OpCodes.Ldarg_0);
        code.Emit(OpCodes.Ldarg_1);
        code.Emit(OpCodes.Stfld, functionField);
        
        code.Emit(OpCodes.Ret);
    }

    // Generate MockInvoke method which redirects the call to the MockDelegate instance.
    private static void GenerateRedirector(TypeBuilder typeBuilder, FieldBuilder functionField, Type delegateType)
    {
        var targetMethod = delegateType.GetMethod("Invoke")!;
        var parameterTypes = targetMethod.GetParameters().Select(parameter => parameter.ParameterType).ToArray();
        var builder = typeBuilder.DefineMethod("MockInvoke",
            MethodAttributes.Public, CallingConventions.Standard,
            targetMethod.ReturnType, parameterTypes);
        
        var code = builder.GetILGenerator();
        
        // Load MockFunction instance.
        code.Emit(OpCodes.Ldarg_0);
        code.Emit(OpCodes.Ldfld, functionField);
        
        // Prepare arguments array.
        code.Emit(OpCodes.Ldc_I4, parameterTypes.Length);
        code.Emit(OpCodes.Newarr, typeof(object));
        for (var index = 0; index < parameterTypes.Length; ++index)
        {
            code.Emit(OpCodes.Dup);
            code.Emit(OpCodes.Ldc_I4, index);
            code.Emit(OpCodes.Ldarg, index + 1);
            if (parameterTypes[index].IsValueType)
                code.Emit(OpCodes.Box, parameterTypes[index]);
            code.Emit(OpCodes.Stelem_Ref);
        }
        
        // Invoke MockDelegate Invoke(...) function.
        code.Emit(OpCodes.Callvirt, typeof(MockDelegate)
            .GetMethod(nameof(MockDelegate.Invoke), [typeof(object?[])])!);
        
        if (targetMethod.ReturnType.IsValueType)
            code.Emit(OpCodes.Unbox_Any, targetMethod.ReturnType);
        
        code.Emit(OpCodes.Ret);
    }

    public static Type GetRedirectorType(Type delegateType) => RedirectorTypes.GetDynamicType(delegateType);
}