using System.Collections.Concurrent;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics.CodeAnalysis;
using System.Reflection;
using System.Runtime.CompilerServices;
using JetBrains.Annotations;
using Robotless.Kernel;
using Robotless.Modules.Annotations;
using Robotless.Modules.Injecting;
using Robotless.Modules.Utilities;

namespace Robotless.Framework;

/// <summary>
/// Entity is the minimal unit of the user-defined objects within a workspace.
/// </summary>
public class Entity : WorkspaceObject, IInjectionProvider
{
    public Entity()
    {
        Components = new ReadOnlyDictionary<Type, Component>(_components);
    }
    
    private readonly ConcurrentDictionary<Type, Component> _components = new();
    
    /// <summary>
    /// Components that attached to this entity.
    /// </summary>
    public IReadOnlyDictionary<Type, Component> Components { get; }
    
    /// <summary>
    /// Attach a component of the specific type to this entity.
    /// </summary>
    /// <param name="type">Component type.</param>
    /// <returns>Component instance.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    public Component AttachComponent(
        [DynamicallyInstantiated] Type type)
    {
        if (!type.IsAssignableTo(typeof(Component)))
            throw new ArgumentException($"Type \"{type.Name}\" is not a component type.", nameof(type));
        return _components.GetOrAdd(type, InstantiateComponent);
        
        Component InstantiateComponent(Type componentType)
        {
            var component = (Component)Workspace.CreateObject(componentType, this);
            InvokeComponentBindEntity(component, this);
            InvokeComponentOnAttach(component);
            return component;
        }
        
        [UnsafeAccessor(UnsafeAccessorKind.Method, Name="set_Entity")]
        static extern void InvokeComponentBindEntity(Component component, Entity entity);
        
        [UnsafeAccessor(UnsafeAccessorKind.Method, Name="OnAttach")]
        static extern void InvokeComponentOnAttach(Component component);
    }

    /// <summary>
    /// Detach the component of the specific type from this entity.
    /// </summary>
    /// <param name="type">Component type.</param>
    /// <returns>True if the component is detached from this entity, false if not found.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    public bool DetachComponent(Type type)
    {
        if (!_components.TryRemove(type, out var component))
            return false;
        InvokeComponentOnDetach(component);
        Workspace.DeleteObject(component.Identifier);
        return true;
        
        [UnsafeAccessor(UnsafeAccessorKind.Method, Name="OnDetach")]
        static extern void InvokeComponentOnDetach(Component component);
    }

    /// <summary>
    /// Get the component instance of the specific type.
    /// </summary>
    /// <param name="type">Component type.</param>
    /// <returns>Component instance, or null if not found.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    public Component? GetComponent(Type type)
        => _components.GetValueOrDefault(type);

    /// <summary>
    /// Get the injection from this entity.
    /// This method can provide components that attached to this entity and this entity as injections,
    /// and it will query the injections of the workspace if the injection is not any of the above.
    /// </summary>
    public virtual object? GetInjection(Type type, object? key, InjectionRequester requester)
    {
        if (!type.IsAssignableTo(typeof(Component)))
            return type.IsInstanceOfType(this) ? this : Workspace.GetInjection(type, key, requester);
        
        var attribute = requester.Member?.GetCustomAttribute<ComponentDependencyAttribute>()?.IsEnsuring;
        return attribute == true ? this.EnsureComponent(type) : GetComponent(type);
    }
    
    /// <summary>
    /// Entity cannot provide any scoped injections, so it will return itself.
    /// </summary>
    /// <returns>This entity.</returns>
    public virtual IInjectionProvider NewScope() => this;
}

public static class EntityExtensions
{
    /// <summary>
    /// Attach a component of the specific type to this entity.
    /// </summary>
    /// <typeparam name="TComponent">Type of the component to attach.</typeparam>
    /// <returns>Component instance.</returns>
    public static TComponent AttachComponent<[DynamicallyInstantiated] TComponent>
        (this Entity entity) where TComponent : Component
        => (TComponent)entity.AttachComponent(typeof(TComponent));
    
    /// <summary>
    /// Detach the component of the specific type from this entity.
    /// </summary>
    /// <typeparam name="TComponent">Type of the component to detach.</typeparam>
    /// <returns>True if the component is detached from this entity, false if not found.</returns>
    public static bool DetachComponent<TComponent>(this Entity entity) where TComponent : Component
        => entity.DetachComponent(typeof(TComponent));
    
    /// <summary>
    /// Get the component instance of the specific type.
    /// </summary>
    /// <typeparam name="TComponent">Type of the component to get.</typeparam>
    /// <returns>Component instance, or null if not found.</returns>
    public static TComponent? GetComponent<TComponent>(this Entity entity) where TComponent : Component
        => (TComponent?)entity.GetComponent(typeof(TComponent));

    /// <summary>
    /// Get the component instance of the specific type,
    /// or throw an exception if the component is not found.
    /// </summary>
    /// <param name="entity">Entity to require component from.</param>
    /// <param name="type">Type of the component to require.</param>
    /// <returns>Component instance.</returns>
    public static Component RequireComponent(this Entity entity, [DynamicallyInstantiated] Type type)
        => entity.GetComponent(type) ??
           throw new Exception($"Cannot find required component \"{type.Name}\".");

    /// <summary>
    /// Get the component instance of the specific type,
    /// or throw an exception if the component is not found.
    /// </summary>
    /// <param name="entity">Entity to require component from.</param>
    /// <typeparam name="TComponent">Type of the component to get.</typeparam>
    /// <returns>Component instance.</returns>
    public static TComponent RequireComponent<[DynamicallyInstantiated] TComponent>
        (this Entity entity) where TComponent : Component
        => (TComponent)entity.RequireComponent(typeof(TComponent));
    
    /// <summary>
    /// Get the component from this entity; if not found, then attach it to this entity.
    /// </summary>
    /// <param name="entity">Entity to get component from or attach component to.</param>
    /// <param name="type">Component type.</param>
    /// <returns>Component instance.</returns>
    public static Component EnsureComponent(this Entity entity, [DynamicallyInstantiated] Type type)
        => entity.GetComponent(type) ?? entity.AttachComponent(type);
    
    /// <summary>
    /// Get the component from this entity; if not found, then attach it to this entity.
    /// </summary>
    /// <typeparam name="TComponent">Component type.</typeparam>
    /// <param name="entity">Entity to get component from or attach component to.</param>
    /// <returns>Component instance.</returns>
    public static TComponent EnsureComponent<[DynamicallyInstantiated] TComponent>
        (this Entity entity)
        where TComponent : Component
        => (TComponent)entity.EnsureComponent(typeof(TComponent));
}