using System.ComponentModel;
using Robotless.Framework;
using Robotless.Modules.Annotations;
using Robotless.Modules.Injecting;

namespace Robotless.Kernel;

public interface IWorkspace : IInjectionProvider
{
    /// <summary>
    /// Identifier for this workspace.
    /// </summary>
    Guid Identifier { get; }

    /// <summary>
    /// Whether if this workspace is running.
    /// </summary>
    bool IsRunning { get; }

    /// <summary>
    /// Create a workspace object in this workspace.
    /// </summary>
    /// <param name="type">
    /// Type of the workspace object to create, must be derived types of <see cref="WorkspaceObject"/>.
    /// </param>
    /// <param name="provider">
    /// Optional injection provider.
    /// If it is null, then this workspace will be used as the injection provider.
    /// </param>
    /// <returns>Workspace object instance.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    WorkspaceObject CreateObject([DynamicallyInstantiated] Type type, IInjectionProvider? provider = null);
    
    /// <summary>
    /// Delete a workspace object from this workspace.
    /// </summary>
    /// <param name="identifier">Identifier for the entity to delete.</param>
    /// <returns>True if the workspace object is deleted, otherwise false.</returns>
    bool DeleteObject(Guid identifier);
    
    /// <summary>
    /// Get the workspace object with the specified identifier from this workspace.
    /// </summary>
    /// <param name="identifier">Identifier for the entity to get.</param>
    /// <returns>Workspace object instance, or null if not found.</returns>
    WorkspaceObject? GetObject(Guid identifier);

    /// <summary>
    /// Get a service from this workspace.
    /// </summary>
    /// <param name="type">Service type, must be assignable to <see cref="IService"/> interface.</param>
    /// <returns>Service instance, or null if not found.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    IService? GetService(Type type);
    
    /// <summary>
    /// Install a service into this workspace.
    /// </summary>
    /// <param name="type">
    /// Service type to install, must be derived types of <see cref="Service"/>.
    /// </param>
    /// <returns>Service instance.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    Service InstallService([DynamicallyInstantiated] Type type);

    /// <summary>
    /// Uninstall a service from this workspace.
    /// </summary>
    /// <param name="type">
    /// Service type to install, must be derived types of <see cref="Service"/>.
    /// </param>
    /// <returns>True if the corresponding service is uninstalled, otherwise false.</returns>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    bool UninstallService(Type type);
}

public static class WorkspaceExtensions
{
    /// <summary>
    /// Create an entity in this workspace.
    /// </summary>
    /// <param name="workspace">Workspace to create entity in.</param>
    /// <typeparam name="TObject">Entity type to create.</typeparam>
    /// <returns>Created entity instance.</returns>
    public static TObject CreateObject<[DynamicallyInstantiated] TObject>(this IWorkspace workspace) 
        where TObject : WorkspaceObject
        => (TObject)workspace.CreateObject(typeof(TObject));

    /// <summary>
    /// Get the workspace object with the specific identifier from this workspace.
    /// </summary>
    /// <param name="workspace">Workspace to get entity from.</param>
    /// <param name="identifier">Identifier of the entity.</param>
    /// <typeparam name="TObject">Entity type.</typeparam>
    /// <returns>Workspace object instance, or null if not found.</returns>
    /// <exception cref="InvalidCastException">
    /// Thrown if the found object cannot be cast to the specified type.
    /// </exception>
    public static TObject? GetObject<TObject>(this IWorkspace workspace, Guid identifier) 
        where TObject : WorkspaceObject
    {
        var entity = workspace.GetObject(identifier);
        if (entity == null)
            return null;
        if (entity is not TObject castedEntity)
            throw new InvalidCastException($"Entity {identifier:B} ({entity.GetType().Name}) " +
                                           $"cannot be casted to type {typeof(TObject).Name}.");
        return castedEntity;
    }

    /// <summary>
    /// Get a service from this workspace.
    /// </summary>
    /// <typeparam name="TService">Service type</typeparam>
    /// <returns>Service instance, or null if not found.</returns>
    public static TService? GetService<TService>(this IWorkspace workspace) where TService : IService
        => (TService?)workspace.GetService(typeof(TService));

    /// <summary>
    /// Install a service into this workspace.
    /// </summary>
    /// <typeparam name="TService">Service type to install.</typeparam>
    /// <returns>Service instance.</returns>
    public static TService InstallService<[DynamicallyInstantiated] TService>(this IWorkspace workspace) where TService : Service
        => (TService)workspace.InstallService(typeof(TService));

    /// <summary>
    /// Uninstall a service from this workspace.
    /// </summary>
    /// <typeparam name="TService">Service type to install.</typeparam>
    /// <returns>True if the corresponding service is uninstalled, otherwise false.</returns>
    public static bool UninstallService<TService>(this IWorkspace workspace) where TService : Service
        => workspace.UninstallService(typeof(TService));
}