﻿using Microsoft.Extensions.Logging;
using Robotless.Modules.Logging;

namespace Robotless.Kernel;

/// <summary>
/// This exception can be converted into a corresponding log item.
/// </summary>
/// <param name="message">Exception message.</param>
/// <param name="data">Details data.</param>
/// <param name="innerException">Inner exception of this exception.</param>
public class PlatformException(string message, Action<dynamic>? data = null, Exception? innerException = null) :
    Exception(message, innerException)
{
    public DateTime Time { get; init; } = DateTime.Now;
    
    public Guid AuthorId { get; init; }
    
    public string? AuthorName { get; init; }
    
    public bool IsCritical { get; init; }

    public Action<dynamic>? Details { get; } = data;
    
    private LogItem? _log;
    
    /// <summary>
    /// Convert this exception to a <see cref="LogItem"/>.
    /// </summary>
    /// <returns>The corresponding log item.</returns>
    public LogItem ToLog()
    {
        if (_log != null)
            return _log;
        _log = new LogItem
        {
            Level = IsCritical ? LogLevel.Critical : LogLevel.Error,
            Time = Time,
            Message = Message,
            AuthorId = AuthorId,
            AuthorName = AuthorName
        };
        Details?.Invoke(_log.Details);
        return _log;
    }
}

public static class PlatformExceptionExtensions
{
    public static PlatformException PlatformException(this LoggerComponent? logger, string message, 
        Action<dynamic>? details = null, Exception? innerException = null)
    {
        var exception = new PlatformException(message, details, innerException);
        logger?.LogException(exception, message, LogLevel.Critical, details);
        return exception;
    }
}