using System.Collections.Concurrent;
using System.Runtime.CompilerServices;
using Robotless.Framework;
using Robotless.Modules.Injecting;
using Robotless.Modules.Injecting.Injectors;

namespace Robotless.Kernel.Utilities;

public class ObjectDirectory(IWorkspace workspace)
{
    
    private readonly ConcurrentDictionary<Guid, Entity> _objects = new();
    
    /// <summary>
    /// Dictionary for entities within this workspace.
    /// </summary>
    public IReadOnlyDictionary<Guid, Entity> Objects => _objects;
    
    public WorkspaceObject CreateObject(Type type, IInjectionProvider? provider = null)
    {
        provider ??= workspace;
        if (!type.IsAssignableTo(typeof(WorkspaceObject)))
            throw new ArgumentException($"Invalid entity type: \"{type.Name}\" is not derived from WorkspaceObject.");
        var instance = (WorkspaceObject)provider.NewObject(type);
        InvokeObjectOnCreate(instance);

        return instance;
        
        [UnsafeAccessor(UnsafeAccessorKind.Method, Name="set_Workspace")]
        static extern void SetObjectWorkspace(WorkspaceObject target, IWorkspace workspace);
        
        [UnsafeAccessor(UnsafeAccessorKind.Method, Name="OnCreate")]
        static extern Task InvokeObjectOnCreate(WorkspaceObject target);
    }

    public bool DeleteObject(Guid identifier)
    {
        if (!_objects.TryRemove(identifier, out var instance))
            return false;
        InvokeObjectOnDelete(instance);
        return true;
        
        [UnsafeAccessor(UnsafeAccessorKind.Method, Name="OnDelete")]
        static extern Task InvokeObjectOnDelete(WorkspaceObject target);
    }
    
    public WorkspaceObject? GetObject(Guid identifier)
        => _objects.GetValueOrDefault(identifier);
}