using System.Collections.Concurrent;
using Robotless.Framework;
using Robotless.Modules.Injecting;

namespace Robotless.Kernel.Utilities;

public class ServiceDirectory(IWorkspace workspace)
{
    private readonly ConcurrentDictionary<Type, Service> _userServices = new();
    
    private readonly ConcurrentDictionary<Type, WorkspaceService> _coreServices = new();

    /// <summary>
    /// Dictionary for user services.
    /// </summary>
    public IReadOnlyDictionary<Type, Service> UserServices => _userServices;
    
    /// <summary>
    /// Dictionary for core services.
    /// </summary>
    public IReadOnlyDictionary<Type, WorkspaceService> CoreServices => _coreServices;
    
    public IService? GetService(Type type)
    {
        if (type.IsAssignableTo(typeof(Service)))
            return _userServices.GetValueOrDefault(type);
        return _coreServices.GetValueOrDefault(type);
    }

    public Service InstallService(Type type)
    {
        if (!type.IsAssignableTo(typeof(Service)))
            throw new ArgumentException($"Invalid service type: \"{type.Name}\" is not derived from Service.");

        var service = (Service)workspace.NewObject(type);

        // Todo
        throw new NotImplementedException();
    }

    public bool UninstallService(Type type)
    {
        if (!type.IsAssignableTo(typeof(Service)))
            throw new ArgumentException($"Invalid service type: \"{type.Name}\" is not derived from Service.");
        
        // Todo
        throw new NotImplementedException();
    }
    
    public WorkspaceService InstallWorkspaceService(Type type)
    {
        if (!type.IsAssignableTo(typeof(WorkspaceService)))
            throw new ArgumentException(
                $"Invalid service workspace type: \"{type.Name}\" is not derived from WorkspaceService.");
        var service = (WorkspaceService)workspace.NewObject(type);
        _coreServices[type] = service;
        return service;
    }

    public WorkspaceService? UninstallWorkspaceService(Type type)
    {
        if (!type.IsAssignableTo(typeof(WorkspaceService)))
            throw new ArgumentException(
                $"Invalid workspace service type: \"{type.Name}\" is not derived from WorkspaceService.");
        return _coreServices.TryRemove(type, out var service) ? service : null;
    }
}