using Robotless.Framework;
using Robotless.Kernel.Utilities;
using Robotless.Modules.Injecting;

namespace Robotless.Kernel;

public class Workspace : IWorkspace
{
    /// <summary>
    /// Check if the platform is running.
    /// </summary>
    public bool IsRunning { get; private set; } = false;

    /// <summary>
    /// Identifier of this workspace.
    /// </summary>
    public Guid Identifier { get; private set; }

    private readonly ObjectDirectory _objects;

    private readonly ServiceDirectory _services;
    
    public Workspace()
    {
        _objects = new ObjectDirectory(this);
        _services = new ServiceDirectory(this);
    }

    public WorkspaceObject CreateObject(Type type, IInjectionProvider? provider = null)
        => _objects.CreateObject(type, provider);

    public bool DeleteObject(Guid identifier)
        => _objects.DeleteObject(identifier);

    public WorkspaceObject? GetObject(Guid identifier)
        => _objects.GetObject(identifier);

    public IService? GetService(Type type) => _services.GetService(type);

    public Service InstallService(Type type) => _services.InstallService(type);

    public bool UninstallService(Type type) => _services.UninstallService(type);

    public Task Lifecycle { get; private set; } = Task.CompletedTask;

    /// <summary>
    /// This method can provide services and this workspace as injections.
    /// </summary>
    public virtual object? GetInjection(Type type, object? key = default, InjectionRequester requester = default)
    {
        if (type.IsAssignableTo(typeof(IService)))
            return GetService(type);
        return type.IsInstanceOfType(this) ? this : null;
    }

    /// <summary>
    /// Workspace cannot provide any scoped injections, so it will return itself.
    /// </summary>
    /// <returns>This Workspace.</returns>
    public virtual IInjectionProvider NewScope() => this;

    /// <summary>
    /// Start the lifecycle of this platform.
    /// </summary>
    public async Task Startup()
    {
        if (IsRunning)
            return;

        IsRunning = true;
    }

    /// <summary>
    /// Stop the lifecycle of this platform.
    /// </summary>
    public async Task Shutdown()
    {
        if (!IsRunning)
            return;

        IsRunning = false;
    }
}