using Robotless.Modules.Annotations;
using Robotless.Modules.Serializing;

namespace Robotless.Kernel;

public class WorkspaceObject : IIdentifiable
{
    internal WorkspaceObject()
    {}
    
    /// <summary>
    /// Identifier for this object.
    /// </summary>
    public Guid Identifier { get; protected set; } = Guid.CreateVersion7();
    
    /// <summary>
    /// Workspace that this object belongs to.
    /// </summary>
    [SnapshotOptions(Ignored = true)]
    public required IWorkspace Workspace { get; init; }
    
    /// <summary>
    /// This method is called after this object is created in a workspace.
    /// Constructors will be invoked every time the workspace startups,
    /// but this method will only be invoked when the object is created in the workspace.
    /// </summary>
    protected virtual Task OnCreate()
    {
        return Task.CompletedTask;
    }

    /// <summary>
    /// This method is called before this object is deleted from a workspace.
    /// Dispose method will be invoked every time the workspace shutdowns,
    /// but this method will only be invoked when the object is deleted from the workspace.
    /// </summary>
    protected virtual Task OnDelete()
    {
        return Task.CompletedTask;
    }
    
    /// <summary>
    /// This method is called after the workspace starts.
    /// </summary>
    [DynamicallyInvoked]
    protected virtual void OnStart() {}
    
    /// <summary>
    /// This method is called before the workspace stops.
    /// </summary>
    [DynamicallyInvoked]
    protected virtual void OnStop() {}
}