using System.Runtime.CompilerServices;
using JetBrains.Annotations;
using Robotless.Framework;
using Robotless.Kernel;
using Robotless.Kernel.Utilities;
using Robotless.Modules.Injecting;

namespace Robotless.Modules.Developing;

/// <summary>
/// A fully customizable workspace without database access, designed for development purposes.
/// By default, no workspace services are installed, users have to install them manually through
/// <see cref="DevelopmentWorkspace.InstallWorkspaceService"/> method.
/// </summary>
public class DevelopmentWorkspace : IWorkspace
{
    /// <summary>
    /// Check if the platform is running.
    /// </summary>
    public bool IsRunning { get; private set; } = false;

    /// <summary>
    /// Identifier of this workspace.
    /// </summary>
    public Guid Identifier { get; private set; } = Guid.CreateVersion7();

    [PublicAPI] public InjectionContainer Injections { get; }

    private readonly ObjectDirectory _objects;

    private readonly ServiceDirectory _services;

    public DevelopmentWorkspace()
    {
        Injections = new InjectionContainer()
        {
            FallbackProviders =
            {
                IInjectionProvider.FromDelegate((type, key, requester) =>
                {
                    if (type.IsAssignableTo(typeof(IService)))
                        return GetService(type);
                    return type.IsInstanceOfType(this) ? this : null;
                }),
                EnsuredComponentProvider.Instance
            }
        };
        _objects = new ObjectDirectory(this);
        _services = new ServiceDirectory(this);
    }

    public WorkspaceObject CreateObject(Type type, IInjectionProvider? provider = null)
        => _objects.CreateObject(type, provider);

    public bool DeleteObject(Guid identifier)
        => _objects.DeleteObject(identifier);

    public WorkspaceObject? GetObject(Guid identifier)
        => _objects.GetObject(identifier);

    public IService? GetService(Type type) => _services.GetService(type);

    public Service InstallService(Type type) => _services.InstallService(type);

    public bool UninstallService(Type type) => _services.UninstallService(type);

    public WorkspaceService InstallWorkspaceService(Type type)
    {
        var service = _services.InstallWorkspaceService(type);
        if (IsRunning)
            StartupWorkspaceService(service);
        return service;
    }

    public bool UninstallWorkspaceService(Type type)
    {
        var service = _services.UninstallWorkspaceService(type);
        if (service == null)
            return false;
        if (IsRunning)
            ShutdownWorkspaceService(service);
        return true;
    }

    /// <summary>
    /// This method can provide services and this workspace as injections.
    /// </summary>
    public object? GetInjection(Type type, object? key = default, InjectionRequester requester = default)
        => Injections.GetInjection(type, key, requester);

    public IInjectionProvider NewScope() => Injections.NewScope();

    public async Task Startup()
    {
        if (IsRunning)
            return;
        IsRunning = true;
        foreach (var (_, service) in _services.CoreServices)
            await StartupWorkspaceService(service);
    }

    public async Task Shutdown()
    {
        if (!IsRunning)
            return;
        IsRunning = false;
        foreach (var (_, service) in _services.CoreServices)
            await ShutdownWorkspaceService(service);
    }

    [UnsafeAccessor(UnsafeAccessorKind.Method, Name = "Startup")]
    private static extern Task StartupWorkspaceService(WorkspaceService service);

    [UnsafeAccessor(UnsafeAccessorKind.Method, Name = "Shutdown")]
    private static extern Task ShutdownWorkspaceService(WorkspaceService service);
}