﻿using System.Reflection;
using System.Text;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Documenting;

/// <summary>
/// This static helper class can format the entry name for specified reflection information.
/// </summary>
public static class EntryName
{
    public static string Get(Type type)
    {
        return $"T:{type.FullName}".Replace('+', '.');
    }

    public static string Get(ConstructorInfo constructor)
        => BuildName('M', constructor, "#ctor").Replace('+', '.');
    
    public static string Get(FieldInfo field)
        => BuildName('F', field).Replace('+', '.');

    public static string Get(PropertyInfo property)
        => BuildName('P', property).Replace('+', '.');

    public static string Get(EventInfo @event)
        => BuildName('E', @event).Replace('+', '.');
    
    public static string Get(MethodInfo method)
    {
        var builder = StringBuilders.Get();
        
        builder.Append("M:");
        if (method.DeclaringType != null)
        {
            builder.Append(method.DeclaringType.FullName?.Replace('+', '.'));
            builder.Append('.');
        }
        builder.Append(method.Name);
        builder.Append('(');
        builder.AppendJoin(',' , method.GetParameters().Select(
            parameter => parameter.ParameterType.FullName?.Replace('+', '.')));
        builder.Append(')');
        
        var name = builder.ToString();
        
        StringBuilders.Return(builder);
        return name;
    }

    private static readonly ObjectPool<StringBuilder> StringBuilders = new()
    {
        Creator = () => new StringBuilder(),
        Cleaner = builder => builder.Clear()
    };
    
    private static string BuildName(char symbol, MemberInfo member, string? memberName = null)
    {
        memberName ??= member.Name;
        var builder = StringBuilders.Get();
        builder.Append(symbol);
        builder.Append(':');
        if (member.DeclaringType != null)
        {
            builder.Append(member.DeclaringType.FullName?.Replace('+', '.'));
            builder.Append('.');
        }
        builder.Append(memberName);
        var name = builder.ToString();
        
        StringBuilders.Return(builder);
        
        return name;
    }
    
    public static string Get(MemberInfo member)
    {
        return member switch
        {
            MethodInfo method => Get(method),
            FieldInfo field => Get(field),
            PropertyInfo property => Get(property),
            EventInfo @event => Get( @event),
            ConstructorInfo constructor => Get(constructor),
            _ => throw new Exception(
                $"Unsupported member type {member.MemberType} for member {member.DeclaringType}.{member.Name}")
        };
    }
}