using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Eventing;

file static class EventTriggerGenerator
{
    private static readonly DynamicTypeCache TriggerTypes = new (GenerateTrigger);
    
    private static Type GenerateTrigger(ModuleBuilder module, Type delegateType)
    {
        var typeBuilder = module.DefineType($"Trigger_{delegateType}", 
            TypeAttributes.Public);
        var fieldEvent = EmitConstructor(typeBuilder, 
            typeof(IEnumerable<>).MakeGenericType(delegateType));
        
        var delegateInvoker = delegateType.GetMethod("Invoke")!;
        var arguments = delegateInvoker.GetParameters()
            .Select(parameter => parameter.ParameterType).ToArray();
        
        var methodBuilder = typeBuilder.DefineMethod("Invoke",
            MethodAttributes.Public, CallingConventions.Standard,
            delegateInvoker.ReturnType,
            delegateInvoker.GetParameters().Select(parameter => parameter.ParameterType).ToArray());
        var code = methodBuilder.GetILGenerator();

        var methodGetEnumerator = typeof(IEnumerable<>).MakeGenericType(delegateType)
            .GetMethod(nameof(IEnumerable<object>.GetEnumerator))!;
        
        code.EmitForEach(delegateType, _ =>
        {
            code.LoadArgument0();
            code.LoadField(fieldEvent);
            code.CallVirtual(methodGetEnumerator);
        }, _ =>
        {
            for (var index = 0; index < arguments.Length; index++)
            {
                code.LoadArgument(index + 1);
            }
            code.Call(delegateInvoker);
            if (delegateInvoker.ReturnType != typeof(void))
                code.Emit(OpCodes.Pop);
        });
        code.MethodReturn();

        return typeBuilder.CreateType();
    }

    private static FieldBuilder EmitConstructor(TypeBuilder typeBuilder, Type eventType)
    {
        var fieldEvent = typeBuilder.DefineField("_event", eventType, 
            FieldAttributes.Private | FieldAttributes.InitOnly);
        var constructorBuilder = typeBuilder.DefineConstructor(
            MethodAttributes.Public, CallingConventions.Standard, [eventType]);
        var constructorCode = constructorBuilder.GetILGenerator();
        constructorCode.LoadArgument0();
        constructorCode.LoadArgument1();
        constructorCode.StoreField(fieldEvent);
        constructorCode.MethodReturn();
        return fieldEvent;
    }

    public static Type GetTriggerType(Type delegateType)
        => TriggerTypes.GetDynamicType(delegateType);
}

public static class EventTrigger<TDelegate> where TDelegate : Delegate
{
    public static TDelegate New(IEnumerable<TDelegate> list)
    {
        var type = EventTriggerGenerator.GetTriggerType(typeof(TDelegate));
        var trigger = Activator.CreateInstance(type, list);
        return type.GetMethod("Invoke")!.CreateDelegate<TDelegate>(trigger);
    }
}