using System.Collections.Concurrent;
using System.Diagnostics.CodeAnalysis;

namespace Robotless.Modules.Eventing;

public class KeyedWeakEvent<TKey, TDelegate> : IKeyedWeakEvent<TKey, TDelegate>
    where TKey : notnull where TDelegate : Delegate
{
    private readonly ConcurrentDictionary<TKey, WeakEvent<TDelegate>> _lists = new();
    
    public void Add(TKey key, object owner, TDelegate handler) 
        => _lists.GetOrAdd(key, _ => new WeakEvent<TDelegate>()).Add(owner, handler);

    public bool Remove(TKey key, object owner, TDelegate handler)
        => _lists.TryGetValue(key, out var group) && group.Remove(owner, handler);

    public bool Remove(TKey key, object owner)
        => _lists.TryGetValue(key, out var group) && group.Remove(owner);

    public bool Remove(object owner)
    {
        var removed = false;
        foreach (var (_, group) in _lists)
        {
            group.Remove(owner);
            removed = true;
        }
        return removed;
    }

    /// <summary>
    /// Clear all handlers.
    /// </summary>
    public void Clear()
        => _lists.Clear();

    /// <summary>
    /// Clear the handlers list of the specified key.
    /// </summary>
    /// <param name="key">Key of the list to remove.</param>
    public void Clear(TKey key)
        => _lists.Remove(key, out _);
    
    /// <summary>
    /// Try to get the event with the specified key.
    /// </summary>
    /// <param name="key">Key for the event to get.</param>
    /// <param name="value">Weak event.</param>
    /// <returns></returns>
    public bool TryGetEvent(TKey key, [NotNullWhen(true)] out WeakEvent<TDelegate>? value)
        => _lists.TryGetValue(key, out value);
    
    /// <summary>
    /// Get the event trigger with the specified key.
    /// </summary>
    /// <param name="key">Event trigger with the specified key, or null if not found.</param>
    public TDelegate? this[TKey key] => _lists.GetValueOrDefault(key)?.Invoke;
}