﻿using System.Collections;
using System.Runtime.CompilerServices;

namespace Robotless.Modules.Eventing;

/// <summary>
/// This event has features different from the standard event: <br/>
/// - Handlers will be automatically removed from this event when their owner is collected by GC. <br/>
/// </summary>
/// <typeparam name="TDelegate">Delegate of event handlers.</typeparam>
public class WeakEvent<TDelegate> : IWeakEvent<TDelegate>, IEnumerable<TDelegate> where TDelegate : Delegate
{
    private readonly ConditionalWeakTable<object, LinkedList<TDelegate>> _handlers = new();
    
    /// <summary>
    /// Delegate to invoke this event.
    /// </summary>
    public TDelegate Invoke { get; }

    public WeakEvent()
    {
        Invoke = EventTrigger<TDelegate>.New(this);
    }

    /// <summary>
    /// Add an event handler to the handler list.
    /// </summary>
    /// <param name="owner">Object which owns this event handler.</param>
    /// <param name="handler">Event handler to add.</param>
    public void Add(object owner, TDelegate handler)
    {
        _handlers.GetOrCreateValue(owner).AddLast(handler);
    }

    /// <summary>
    /// Remove an event handler from the handler list.
    /// </summary>
    /// <param name="owner">Object which owns this event handler.</param>
    /// <param name="handler">Event handler to remove.</param>
    /// <returns>
    /// True if a handler is removed, otherwise false.
    /// </returns>
    public bool Remove(object owner, TDelegate handler)
    {
        return _handlers.TryGetValue(owner, out var handlers) && handlers.Remove(handler);
    }

    /// <summary>
    /// Remove all event handlers owned by the specified object.
    /// </summary>
    /// <param name="owner">Object whose event handlers will be removed.</param>
    /// <returns>
    /// True if any handler of the specified object is removed, otherwise false.
    /// </returns>
    public bool Remove(object owner)
    {
        return _handlers.Remove(owner);
    }

    /// <summary>
    /// Clear all handlers in the handler list.
    /// </summary>
    public void Clear()
        => _handlers.Clear();

    public IEnumerator<TDelegate> GetEnumerator()
        => _handlers.SelectMany(pair => pair.Value).GetEnumerator();

    IEnumerator<TDelegate> IEnumerable<TDelegate>.GetEnumerator()
        => GetEnumerator();

    IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();
}