using System.ComponentModel;

namespace Robotless.Modules.Injecting;

public interface IInjectionContainer : IInjectionProvider
{
    /// <summary>
    /// Add a type to this container.
    /// This implementation type will be instantiated and injected when the corresponding type is requested.
    /// </summary>
    /// <param name="type">Type that this implementation is bound to.</param>
    /// <param name="implementation">Implementation type to add.</param>
    /// <param name="lifespan">Lifespan of instances instantiated from this implementation type.</param>
    /// <param name="key">Optional key for the injection.</param>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    void AddInjection(Type type, Type implementation, InjectionLifespan lifespan, 
        object? key = null);

    /// <summary>
    /// Add a factory to this container.
    /// This factory method will be invoked when the corresponding type is requested.
    /// </summary>
    /// <param name="type">Type that this factory is bound to.</param>
    /// <param name="factory">Factory to add.</param>
    /// <param name="lifespan">Lifespan of instances returned by this factory.</param>
    /// <param name="key">Optional key for the injection.</param>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    void AddInjection(Type type, Func<IInjectionProvider, InjectionRequester, object> factory, InjectionLifespan lifespan, 
        object? key = null);

    /// <summary>
    /// Add a value to this container.
    /// This value will be returned when the corresponding type is requested.
    /// </summary>
    /// <param name="type">Type that this value is bound to.</param>
    /// <param name="value">Value to add.</param>
    /// <param name="key">Optional key for the injection.</param>
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    void AddInjection(Type type, object value, object? key = null);

    /// <summary>
    /// Add a redirection entry into this container.
    /// The request to injections of the 'fromType' with 'fromKey'
    /// will be considered as the request to injections of the 'toType' with 'toKey'. <br/>
    /// This redirection can also be removed by <see cref="RemoveInjection"/> method
    /// with 'fromType' and 'fromKey' as arguments.
    /// </summary>
    /// <remarks>
    /// This redirection will always be in effect, even if it is added before the entry to redirect to is added. <br/>
    /// The entry of 'fromType' with 'fromKey' will be replaced by this redirection entry,
    /// this operation is not reversible. <br/>
    /// </remarks>
    /// <param name="fromType">Type to redirect from.</param>
    /// <param name="fromKey">Key to redirect from.</param>
    /// <param name="toType">Type to redirect to.</param>
    /// <param name="toKey">Key to redirect to.</param>
    void AddRedirection(Type fromType, object? fromKey, Type toType, object? toKey);

    /// <summary>
    /// Try to remove a resource from the container.
    /// </summary>
    /// <param name="type">Type of the resource to remove.</param>
    /// <param name="key">Optional key for the injection.</param>
    /// <returns>True if the resource is removed from this container, false if it is not found.</returns>
    bool RemoveInjection(Type type, object? key = null);
}

public static class ResourceContainerExtensions
{
    public static IInjectionContainer AddTransient(this IInjectionContainer container, Type type, Type implementation,
        object? key = null)
    {
        container.AddInjection(type, implementation, InjectionLifespan.Transient, key);
        return container;
    }
    
    public static IInjectionContainer AddTransient<TImplementation>(this IInjectionContainer container, 
        object? key = null)
    {
        container.AddInjection(typeof(TImplementation), typeof(TImplementation), InjectionLifespan.Transient, key);
        return container;
    }

    public static IInjectionContainer AddTransient<TInjection>(this IInjectionContainer container, 
        Func<IInjectionProvider, InjectionRequester, TInjection> factory, object? key = null) where TInjection : class
    {
        container.AddInjection(typeof(TInjection), factory, InjectionLifespan.Transient, key);
        return container;
    }

    public static IInjectionContainer AddSingleton(this IInjectionContainer container, Type type, Type implementation,
        object? key = null)
    {
        container.AddInjection(type, implementation, InjectionLifespan.Singleton, key);
        return container;
    }
    
    public static IInjectionContainer AddSingleton<TImplementation>(this IInjectionContainer container, 
        object? key = null)
    {
        container.AddInjection(typeof(TImplementation), typeof(TImplementation), InjectionLifespan.Singleton, key);
        return container;
    }

    public static IInjectionContainer AddSingleton<TInjection>(this IInjectionContainer container,
        Func<IInjectionProvider, InjectionRequester, TInjection> factory, object? key = null) where TInjection : class
    {
        container.AddInjection(typeof(TInjection), factory, InjectionLifespan.Singleton, key);
        return container;
    }
    
    public static IInjectionContainer AddConstant<TInjection>(this IInjectionContainer container, 
        TInjection value, object? key = null)
        where TInjection : notnull
    {
        container.AddInjection(typeof(TInjection), value, key);
        return container;
    }

    public static IInjectionContainer AddRedirection<TFrom, TTo>(this IInjectionContainer container,
        object? fromKey = null, object? toKey = null)
        where TTo : TFrom
    {
        container.AddRedirection(typeof(TFrom), fromKey, typeof(TTo), toKey);
        return container;
    }

    public static IInjectionContainer RemoveInjection<TInjection>(this IInjectionContainer container, 
        object? key = null)
    {
        container.RemoveInjection(typeof(TInjection), key);
        return container;
    }
}