using System.ComponentModel;
using Robotless.Modules.Injecting.Injectors;

namespace Robotless.Modules.Injecting;

public interface IInjectionProvider
{
    /// <summary>
    /// Get a resource of the specified category for this provider.
    /// </summary>
    /// <param name="type">Category type of the resource.</param>
    /// <param name="key">Optional key for the requested injection.</param>
    /// <param name="requester">Optional information about the target requested this injection.</param>
    /// <returns>Requested resource, or null if not found.</returns>
    object? GetInjection(Type type, object? key = default, InjectionRequester requester = default);

    /// <summary>
    /// Create a new injection scope for this provider.
    /// </summary>
    /// <returns>Provider for the new injection scope.</returns>
    IInjectionProvider NewScope();

    /// <summary>
    /// Delegate that has the same signature with <see cref="IInjectionProvider.GetInjection"/> method.
    /// </summary>
    /// <param name="type">Category type of the resource.</param>
    /// <param name="key">Optional key for the requested injection.</param>
    /// <param name="requester">Optional information about the target requested this injection.</param>
    /// <returns>Requested resource, or null if not found.</returns>
    public delegate object? ProviderDelegate(Type type, object? key = default, InjectionRequester requester = default);

    /// <summary>
    /// Create a new instance of injection provider from a delegate.
    /// </summary>
    /// <param name="functor">Delegate to wrap into an injection provider.</param>
    /// <returns>Injection provider created from the specified delegate.</returns>
    public static IInjectionProvider FromDelegate(ProviderDelegate functor)
        => new FunctorInjectionProvider(functor);

    /// <summary>
    /// This class wraps a method delegate into an injection provider.
    /// </summary>
    /// <param name="functor">Functor to wrap.</param>
    private class FunctorInjectionProvider(ProviderDelegate functor)
        : IInjectionProvider
    {
        public object? GetInjection(Type type, object? key = default, InjectionRequester requester = default)
            => functor(type, key, requester);

        public IInjectionProvider NewScope() => this;
    }
}

public static class ResourceProviderExtensions
{
    /// <summary>
    /// Get a resource of the specified category for this provider.
    /// </summary>
    /// <typeparam name="TObject">Category type of the resource.</typeparam>
    /// <param name="provider">Provider to get the object from.</param>
    /// <param name="key">Optional key for the requested injection.</param>
    /// <returns>Requested resource, or null if not found.</returns>
    public static TObject? GetInjection<TObject>(this IInjectionProvider provider, object? key = null)
        => (TObject?)provider.GetInjection(typeof(TObject), key);

    public static object RequireInjection(this IInjectionProvider provider, Type type, object? key = null)
        => provider.GetInjection(type, key) ??
           throw new Exception($"Cannot find required injection '{type.Name}' with key '{key}'");
    
    public static TObject RequireInjection<TObject>(this IInjectionProvider provider, object? key = null)
        => (TObject?)provider.GetInjection(typeof(TObject), key) ??
           throw new Exception($"Cannot find required injection '{typeof(TObject).Name}' with key '{key}'");
    
    [EditorBrowsable(EditorBrowsableState.Advanced)]
    public static void NewObject(this IInjectionProvider provider, object target)
    {
        var type = target.GetType();
        if (!ConstructorInjector.Get(type).TryInject(target, provider, out var missing) ||
            !MemberInjector.Get(type).TryInject(target, provider, out missing))
            throw new InjectionFailureException(type, provider, missing);
    }
    
    /// <summary>
    /// Instantiate a new object of the specified type and inject all required dependencies.
    /// </summary>
    /// <param name="provider">Provider to get injections from.</param>
    /// <param name="type">Type to instantiate.</param>
    /// <returns>Instantiated object instance.</returns>
    /// <exception cref="InjectionFailureException">
    /// Throw if any required injections cannot be found within the specified provider.
    /// </exception>
    public static object NewObject(this IInjectionProvider provider, Type type)
    {
        if (!ConstructorInjector.Get(type).TryInject(out var instance, provider, out var missing) ||
            !MemberInjector.Get(type).TryInject(instance, provider, out missing))
            throw new InjectionFailureException(type, provider, missing);
        return instance;
    }
    
    /// <summary>
    /// Instantiate a new object of the specified type and inject all required dependencies.
    /// </summary>
    /// <typeparam name="TObject">Type to instantiate.</typeparam>
    /// <param name="provider">Provider to get injections from.</param>
    /// <returns>Instantiated object instance.</returns>
    /// <exception cref="InjectionFailureException">
    /// Throw if any required injections cannot be found within the specified provider.
    /// </exception>
    public static TObject NewObject<TObject>(this IInjectionProvider provider)
        => (TObject)provider.NewObject(typeof(TObject));
    
    /// <summary>
    /// Inject the members of this object with the specified injection provider.
    /// </summary>
    /// <param name="target">Object whose members will be injected.</param>
    /// <param name="provider">Provider to get injections from.</param>
    /// <typeparam name="TTarget">Type of this object.</typeparam>
    /// <returns>This object.</returns>
    /// <exception cref="InjectionFailureException">
    /// Throw if any required injections cannot be found within the specified provider.
    /// </exception>
    public static TTarget InjectedWith<TTarget>(this TTarget target, IInjectionProvider provider)
        where TTarget : notnull
    {
        if (!MemberInjector.Get(typeof(TTarget)).TryInject(target, provider, out var missing))
            throw new InjectionFailureException(typeof(TTarget), provider, missing);
        return target;
    }
}