using System.Diagnostics.CodeAnalysis;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Injecting;

public partial class InjectionContainer
{
    public class InjectionScope(
        InjectionContainer container,
        InjectionScope? parent = null) : IInjectionProvider
    {
        private readonly ConcurrentKeyedDictionary<Type, object, object> _scoped = new();

        /// <summary>
        /// Search the cached scoped injections from this scope and its chained parent scopes.
        /// </summary>
        /// <returns>
        /// True if this scope or ancestors hold the cached scoped instance for the required injection;
        /// otherwise false.
        /// </returns>
        private bool TryGetScopedInjection(Type type, object key, [NotNullWhen(true)] out object? value)
        {
            if (_scoped.TryGetValue(type, key, out value))
                return true;
            return parent != null && parent.TryGetScopedInjection(type, key, out value);
        }
        
        public object? GetInjection(Type type, object? key = null, InjectionRequester requester = default)
        {
            key ??= NullKey.Value;
            // Search in the cache for scoped injections.
            if (TryGetScopedInjection(type, key, out var value))
                return value;
            
            if (!container.TryGetInjection(type, key, requester, out var entry, out value))
                return null;
            if (entry?.Lifespan == InjectionLifespan.Scoped)
                _scoped.SetValue(type, key, value);
            return value;
        }

        /// <summary>
        /// Create a new nested scope of this scope.
        /// </summary>
        /// <returns>New nested sub-scope of current scope.</returns>
        public IInjectionProvider NewScope() => new InjectionScope(container, this);
    }
}