using System.Diagnostics.CodeAnalysis;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Injecting;

public partial class InjectionContainer : IInjectionContainer
{
    /// <summary>
    /// The default key for injections when optional key is null.
    /// </summary>
    private sealed record NullKey
    {
        private NullKey()
        {
        }

        public static readonly NullKey Value = new();
    }

    private interface IInjectionEntry
    {
        InjectionLifespan Lifespan { get; }
    }

    /// <summary>
    /// Fallback providers to query if this container cannot provide the requested injection.
    /// </summary>
    public ISet<IInjectionProvider> FallbackProviders { get; } = new HashSet<IInjectionProvider>();

    private readonly ConcurrentKeyedDictionary<Type, object, IInjectionEntry> _entries = new();

    private readonly ConcurrentKeyedDictionary<Type, object, object> _singletons = new();

    /// <summary>
    /// Search the entry for the requested injection and instantiate it.
    /// If the entry is a singleton, then the injection will be cached in this container.
    /// </summary>
    /// <returns>True if the injection is found and instantiated, otherwise false.</returns>
    private bool TryGetInjection(Type type, object key, InjectionRequester requester, out IInjectionEntry? entry,
        [NotNullWhen(true)] out object? value)
    {
        entry = null;
        value = null;

        if (_singletons.TryGetValue(type, key, out value))
            return true;

        if (!_entries.TryGetValue(type, key, out entry))
        {
            value = FallbackProviders
                .Select(provider => provider.GetInjection(type, key, requester))
                .FirstOrDefault(injection => injection != null);
            return value != null;
        }

        value = entry switch
        {
            ConstantInjectionEntry constantEntry => constantEntry.Value,
            FactoryInjectionEntry factoryEntry => factoryEntry.Factory(this, requester),
            TypeInjectionEntry typeEntry => this.NewObject(typeEntry.Implementation),
            RedirectionEntry redirectionEntry => 
                GetInjection(redirectionEntry.TargetType, key, requester),
            _ => throw new Exception($"Unsupported injection entry type \"{entry.GetType().Name}\".")
        };

        if (value == null)
            return false;

        if (entry.Lifespan == InjectionLifespan.Singleton)
            _singletons.SetValue(type, key, value);
        return true;
    }

    /// <summary>
    /// Get an injection from this container.
    /// </summary>
    /// <param name="type">Type of the injection.</param>
    /// <param name="key">The key for the requested injection.</param>
    /// <param name="requester">This parameter is ignored.</param>
    /// <returns>Injection instance, or null if not found.</returns>
    public object? GetInjection(Type type, object? key = null, InjectionRequester requester = default)
        => TryGetInjection(type, key ?? NullKey.Value, requester, out _, out var value) ? value : null;

    public IInjectionProvider NewScope() => new InjectionScope(this);

    public void AddInjection(Type type, Type implementation, InjectionLifespan lifespan, object? key = null)
        => _entries.SetValue(type, key ?? NullKey.Value,
            new TypeInjectionEntry(implementation) { Lifespan = lifespan });

    public void AddInjection(Type type, Func<IInjectionProvider, InjectionRequester, object> factory,
        InjectionLifespan lifespan,
        object? key = null)
        => _entries.SetValue(type, key ?? NullKey.Value,
            new FactoryInjectionEntry(factory) { Lifespan = lifespan });

    public void AddInjection(Type type, object value, object? key = null)
        => _entries.SetValue(type, key ?? NullKey.Value,
            new ConstantInjectionEntry(value));

    public void AddRedirection(Type fromType, object? fromKey, Type toType, object? toKey)
        => _entries.SetValue(fromType, fromKey ?? NullKey.Value,
            new RedirectionEntry(toType, toKey));

    public bool RemoveInjection(Type type, object? key = null)
        => _entries.Remove(type, key ?? NullKey.Value);
}