using System.Diagnostics.CodeAnalysis;
using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using JetBrains.Annotations;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Injecting.Injectors;

[RequiresDynamicCode("`System.Reflection.Emit` is used in this class.")]
public sealed class ConstructorInjector
{
    private static readonly ConditionalWeakTable<Type, ConstructorInjector> Injectors = new();

    public static ConstructorInjector Get(Type type)
    {
        if (Injectors.TryGetValue(type, out var injector))
            return injector;
        injector = new ConstructorInjector(type);
        Injectors.Add(type, injector);
        return injector;
    }

    /// <summary>
    /// Try to instantiate an instance of the specific type with the given provider.
    /// </summary>
    /// <param name="target">Instantiated instance.</param>
    /// <param name="provider">Provider to get injections from.</param>
    /// <param name="missing">
    /// The requester whose injection cannot be found from the provider.
    /// It will be the default value if this method returns true.
    /// </param>
    /// <typeparam name="TTarget">Target type to instantiate.</typeparam>
    /// <returns>
    /// True if the type has been successfully instantiated,
    /// false if any injection requirement cannot be satisfied.
    /// </returns>
    [PublicAPI]
    public static bool TryInject<TTarget>([NotNullWhen(true)] out TTarget? target,
        IInjectionProvider provider, out InjectionRequester missing)
    {
        if (Get(typeof(TTarget)).TryInject(out var instance, provider, out missing))
        {
            target = (TTarget)instance;
            return true;
        }

        target = default;
        return false;
    }

    /// <summary>
    /// Try to instantiate the instance with the given provider.
    /// </summary>
    /// <param name="target">Uninitialized instance.</param>
    /// <param name="provider">Provider to get injections from.</param>
    /// <param name="missing">
    /// The requester whose injection cannot be found from the provider.
    /// It will be the default value if this method returns true.
    /// </param>
    /// <returns>
    /// True if the type has been successfully instantiated,
    /// false if any injection requirement cannot be satisfied.
    /// </returns>
    [PublicAPI]
    public bool TryInject(object target, IInjectionProvider provider, out InjectionRequester missing)
    {
        var requester = _injector(target, provider);
        if (requester != null)
        {
            missing = requester.Value;
            return false;
        }

        missing = default;
        return true;
    }

    /// <summary>
    /// Try to instantiate an instance of the specific type with the given provider.
    /// </summary>
    /// <param name="target">Instantiated instance.</param>
    /// <param name="provider">Provider to get injections from.</param>
    /// <param name="missing">
    /// The requester whose injection cannot be found from the provider.
    /// It will be the default value if this method returns true.
    /// </param>
    /// <returns>
    /// True if the type has been successfully instantiated,
    /// false if any injection requirement cannot be satisfied.
    /// </returns>
    [PublicAPI]
    public bool TryInject([NotNullWhen(true)] out object? target, 
        IInjectionProvider provider, out InjectionRequester missing)
    {
        target = RuntimeHelpers.GetUninitializedObject(_type);
        return TryInject(target, provider, out missing);
    }

    private readonly Type _type;

    private readonly Func<object, IInjectionProvider, InjectionRequester?> _injector;

    private ConstructorInjector(Type type)
    {
        if (type.IsPrimitive || type == typeof(string))
            throw new InvalidOperationException($"Cannot inject primitive type or string \"{type.Name}\".");
        if (type.IsGenericTypeDefinition)
            throw new InvalidOperationException($"Cannot inject generic type definition \"{type.Name}\".");
        
        _type = type;

        var builder = new DynamicMethod(
            $"ConstructorInjector_{type.Name}",
            MethodAttributes.Static | MethodAttributes.Public,
            CallingConventions.Standard, typeof(InjectionRequester?),
            [typeof(object), typeof(IInjectionProvider)], type, true);

        var code = builder.GetILGenerator();

        var constructors = type
            .GetConstructors(BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic)
            .OrderBy(constructor => constructor.GetParameters().Length).ToList();

        var maxVariableCount = constructors[^1].GetParameters().Length;

        var injectionVariables = new List<LocalBuilder>();
        for (var index = 0; index < maxVariableCount; ++index)
        {
            injectionVariables.Add(code.DeclareLocal(typeof(object)));
        }

        var context = new EmittingContext(type, code)
        {
            InjectionVariables = injectionVariables,
        };

        foreach (var constructor in constructors)
        {
            EmitTryConstructor(ref context, constructor);
        }

        if (type.IsValueType)
        {
            code.LoadArgument0();
            code.Call(typeof(Unsafe).GetMethod("Unbox")!.MakeGenericMethod(type));
            code.LoadLocal(context.VariableTarget);
            code.Emit(OpCodes.Stobj, type);
        }
        
        code.LoadLocal(context.VariableRequester);
        code.NewObject(typeof(InjectionRequester?).GetConstructor([typeof(InjectionRequester)])!);
        code.MethodReturn();

        _injector = builder.CreateDelegate<Func<object, IInjectionProvider, InjectionRequester?>>();
    }

    private readonly struct EmittingContext
    {
        public ILGenerator Code { get; }
        public Type Type { get; }
        public LocalBuilder VariableTarget { get; }
        public required List<LocalBuilder> InjectionVariables { get; init; } 
        public LocalBuilder VariableRequester { get; }
        public LocalBuilder VariableParameters { get; }
        
        public LocalBuilder VariableMissingRequester { get; }

        public EmittingContext(Type type, ILGenerator code)
        {
            Code = code;
            Type = type;
            VariableTarget = code.DeclareLocal(type);
            
            VariableRequester = code.DeclareLocal(typeof(InjectionRequester));
            VariableParameters = code.DeclareLocal(typeof(ParameterInfo[]));
            VariableMissingRequester = code.DeclareLocal(typeof(InjectionRequester?));
            
            InjectionVariables = (List<LocalBuilder>) [];
            
            code.LoadArgument0();
            if (type.IsValueType)
                code.Emit(OpCodes.Unbox_Any, type);
            code.StoreLocal(VariableTarget);
        }
        
        public void EmitLoadTarget()
        {
            if (Type.IsValueType)
                Code.LoadLocalAddress(VariableTarget);
            else
                Code.LoadLocal(VariableTarget);
        }
    }

    private static void EmitTryConstructor(ref EmittingContext context, ConstructorInfo constructor)
    {
        var code = context.Code;

        code.EmitConstructorInfo(constructor);
        code.Emit(OpCodes.Callvirt, typeof(MethodBase).GetMethod(nameof(MethodBase.GetParameters))!);
        code.StoreLocal(context.VariableParameters);
        
        var labelFailed = code.DefineLabel();

        var parameters = constructor.GetParameters();
        for (var index = 0; index < parameters.Length; ++index)
        {
            var parameter = parameters[index];
            var attribute = parameter.GetCustomAttribute<InjectionAttribute>();
            var injectionType = attribute?.Type ?? parameter.ParameterType;
            var variableInjection = context.InjectionVariables[index];
            
            EmitGettingInjection(ref context, injectionType, index, attribute?.Key);
            // Load the injection to check if it is null.
            code.LoadLocal(variableInjection);
            if (!parameter.HasDefaultValue)
                code.BreakIfFalse(labelFailed);
            else
            {
                var labelNext = code.DefineLabel();
                code.BreakIfTrue(labelNext);
                code.LoadParameterDefaultValue(parameter);
                if (injectionType.IsValueType)
                    code.Emit(OpCodes.Box, injectionType);
                code.StoreLocal(variableInjection);
                code.MarkLabel(labelNext);
            }
        }

        // Load target object.
        context.EmitLoadTarget();
        // Load arguments.
        for (var index = 0; index < parameters.Length; ++index)
        {
            code.LoadLocal(context.InjectionVariables[index]);
            if (parameters[index].ParameterType.IsValueType)
                code.Emit(OpCodes.Unbox_Any, parameters[index].ParameterType);
        }

        // Invoke the constructor.
        code.Call(constructor);

        if (context.Type.IsValueType)
        {
            code.LoadArgument0();
            code.Call(typeof(Unsafe).GetMethod("Unbox")!.MakeGenericMethod(context.Type));
            code.LoadLocal(context.VariableTarget);
            code.Emit(OpCodes.Stobj, context.Type);
        }

        code.LoadLocalAddress(context.VariableMissingRequester);
        code.Emit(OpCodes.Initobj, typeof(InjectionRequester?));
        code.LoadLocal(context.VariableMissingRequester);
        code.MethodReturn();

        code.MarkLabel(labelFailed);
    }

    // Generate code for getting injection from the source.
    private static void EmitGettingInjection(ref EmittingContext context,
        Type category, int parameterIndex, object? key)
    {
        var code = context.Code;

        // Load injection source.
        code.Emit(OpCodes.Ldarg_1);

        // Load injection type.
        code.EmitTypeInfo(category);

        // Load injection key.
        code.LoadBoxedLiteral(key);

        // Load injection target.
        code.LoadLocalAddress(context.VariableRequester);
        code.LoadArgument0();
        code.LoadLocal(context.VariableParameters);
        code.LoadLiteral(parameterIndex);
        code.Emit(OpCodes.Ldelem_Ref);
        code.LoadNull();
        code.Call(typeof(InjectionRequester).GetConstructor(
            [typeof(object), typeof(ParameterInfo), typeof(MemberInfo)])!);
        code.LoadLocal(context.VariableRequester);
        
        // Query the container for specific injection.
        code.Emit(OpCodes.Callvirt,
            typeof(IInjectionProvider).GetMethod(nameof(IInjectionProvider.GetInjection))!);

        // Store the injection to the variable.
        code.Emit(OpCodes.Stloc, context.InjectionVariables[parameterIndex]);
    }
}