using System.Diagnostics.CodeAnalysis;
using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using Robotless.Modules.Utilities;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Injecting.Injectors;

[RequiresDynamicCode("`System.Reflection.Emit` is used in this class.")]
public sealed class MemberInjector
{
    private static readonly ConditionalWeakTable<Type, MemberInjector> Injectors = new();

    public static MemberInjector Get(Type type)
    {
        if (Injectors.TryGetValue(type, out var injector))
            return injector;
        injector = new MemberInjector(type);
        Injectors.Add(type, injector);
        return injector;
    }

    public static bool TryInject<TTarget>(object target, IInjectionProvider provider, out InjectionRequester missing)
        where TTarget : notnull
        => Get(typeof(TTarget)).TryInject(target, provider, out missing);

    /// <summary>
    /// Inject the members of the specified object.
    /// If the specified object is a boxed struct, then the boxed value will be replaced.
    /// </summary>
    /// <param name="target">Target object, can be boxed structs.</param>
    /// <param name="provider">Provider to get injections from.</param>
    /// <param name="missing">
    /// The requester whose injection requirement cannot be satisfied.
    /// It will be the default value if this method returns true.
    /// </param>
    /// <returns>
    /// True if all required injections of the specified object are found and injected,
    /// otherwise false.
    /// </returns>
    public bool TryInject(object target, IInjectionProvider provider, out InjectionRequester missing)
    {
        var requester = _injector(target, provider);
        if (requester != null)
        {
            missing = requester.Value;
            return false;
        }

        missing = default;
        return true;
    }

    /// <summary>
    /// Try to update the injections of the specified object.
    /// </summary>
    /// <param name="target">
    /// Object whose members with specified type of injections will be updated.
    /// </param>
    /// <param name="type">Injection type.</param>
    /// <param name="injection">Injection instance.</param>
    /// <returns>
    /// True if the members with specified type of injections are updated;
    /// false if no member is injected with the specified type.
    /// </returns>
    public bool TryUpdate(object target, Type type, object injection)
    {
        if (!_injections.TryGetValues(type, out var members))
            return false;
        foreach (var member in members)
        {
            switch (member)
            {
                case FieldInfo field:
                    field.SetValue(target, injection);
                    break;
                case PropertyInfo property:
                    property.SetValue(target, injection);
                    break;
                default:
                    throw new Exception("Unsupported injection member type \"{member.MemberType}\".");
            }
        }

        return true;
    }

    private readonly Func<object, IInjectionProvider, InjectionRequester?> _injector;

    private readonly MultiDictionary<Type, MemberInfo> _injections = new();

    private MemberInjector(Type type)
    {
        if (type.IsPrimitive || type == typeof(string))
            throw new InvalidOperationException("Cannot inject primitive types or strings.");

        var builder = new DynamicMethod($"MemberInjector_{type.Name}",
            MethodAttributes.Static | MethodAttributes.Public, CallingConventions.Standard,
            typeof(InjectionRequester?), [typeof(object), typeof(IInjectionProvider)],
            type, true);

        var code = builder.GetILGenerator();

        // Local variable to store the injection.
        var context = new EmittingContext(type, code);

        code.LoadArgument0();
        if (type.IsValueType)
            code.Emit(OpCodes.Unbox_Any, type);
        code.StoreLocal(context.VariableTarget);

        var labelFailed = code.DefineLabel();

        foreach (var member in type.GetMembers(
                         BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic)
                     .Where(IsInjectionTarget))
        {
            var attribute = member.GetCustomAttribute<InjectionAttribute>();
            var required = attribute?.IsRequired ?? true;
            // ↑ If attribute is null, then this member must be selected for 'required' keyword.
            var labelContinue = code.DefineLabel();
            var injectionType = attribute?.Type;
            switch (member)
            {
                case FieldInfo field:
                    injectionType ??= field.FieldType;
                    EmitGettingInjection(context, injectionType, attribute?.Key, field);
                    code.If(_ => code.LoadLocal(context.VariableInjection),
                        whenFalse: required
                            ? _ => code.Break(labelFailed)
                            : _ => code.Break(labelContinue));
                    EmitInjectField(context, field);
                    break;
                case PropertyInfo property:
                    injectionType ??= property.PropertyType;
                    EmitGettingInjection(context, injectionType, attribute?.Key, property);
                    code.If(_ => code.LoadLocal(context.VariableInjection),
                        whenFalse: required
                            ? _ => code.Break(labelFailed)
                            : _ => code.Break(labelContinue));
                    EmitInjectProperty(context, property);
                    break;
                default:
                    throw new Exception($"Unsupported injecting member type {member.MemberType}.");
            }

            _injections.Add(injectionType, member);

            code.MarkLabel(labelContinue);
        }

        var labelReturn = code.DefineLabel();

        if (type.IsValueType)
        {
            code.LoadArgument0();
            code.Call(typeof(Unsafe).GetMethod("Unbox")!.MakeGenericMethod(type));
            code.LoadLocal(context.VariableTarget);
            code.Emit(OpCodes.Stobj, type);
        }

        // Construct a null missing requester.
        code.LoadLocalAddress(context.VariableMissingRequester);
        code.Emit(OpCodes.Initobj, typeof(InjectionRequester?));
        code.LoadLocal(context.VariableMissingRequester);
        
        code.Break(labelReturn);

        code.MarkLabel(labelFailed);
        
        // Construct a missing requester with the last requester.
        code.LoadLocal(context.VariableRequester);
        code.NewObject(typeof(InjectionRequester?).GetConstructor([typeof(InjectionRequester)])!);

        code.MarkLabel(labelReturn);
        code.MethodReturn();

        _injector = builder.CreateDelegate<Func<object, IInjectionProvider, InjectionRequester?>>();
    }

    private static bool IsInjectionTarget(MemberInfo member)
    {
        switch (member)
        {
            case FieldInfo { IsLiteral: false, IsInitOnly: false }:
                break;
            case PropertyInfo { CanWrite: true }:
                break;
            default:
                return false;
        }

        return member.IsDefined(typeof(InjectionAttribute)) || member.IsDefined(typeof(RequiredMemberAttribute));
    }

    private readonly struct EmittingContext(Type type, ILGenerator code)
    {
        public ILGenerator Code { get; } = code;
        public Type Type { get; } = type;
        public LocalBuilder VariableTarget { get; } = code.DeclareLocal(type);
        public LocalBuilder VariableInjection { get; } = code.DeclareLocal(typeof(object));
        public LocalBuilder VariableRequester { get; } = code.DeclareLocal(typeof(InjectionRequester));

        public LocalBuilder VariableMissingRequester { get; } = code.DeclareLocal(typeof(InjectionRequester?));
        
        public void EmitLoadTarget()
        {
            if (Type.IsValueType)
                Code.LoadLocalAddress(VariableTarget);
            else
                Code.LoadLocal(VariableTarget);
        }
    }

    private static void EmitGettingInjection(in EmittingContext context, Type type, object? key, 
        MemberInfo requester)
    {
        var code = context.Code;

        // Load injection source.
        code.Emit(OpCodes.Ldarg_1);
        code.EmitTypeInfo(type);
        code.LoadBoxedLiteral(key);
        
        // Load injection target.
        code.LoadLocalAddress(context.VariableRequester);
        code.LoadArgument0();
        code.LoadNull();
        switch (requester)
        {
            case FieldInfo field:
                code.EmitFieldInfo(field);
                break;
            case PropertyInfo property:
                code.EmitPropertyInfo(property);
                break;
            default:
                throw new Exception("Unsupported requester type.");
        }
        code.Call(typeof(InjectionRequester).GetConstructor(
            [typeof(object), typeof(ParameterInfo), typeof(MemberInfo)])!);
        code.LoadLocal(context.VariableRequester);
        
        // Query the container for specific injection.
        code.Emit(OpCodes.Callvirt,
            typeof(IInjectionProvider).GetMethod(nameof(IInjectionProvider.GetInjection))!);

        // Store the injection to the variable.
        code.StoreLocal(context.VariableInjection);
    }

    private static void EmitInjectField(in EmittingContext context, FieldInfo field)
    {
        context.EmitLoadTarget();
        context.Code.LoadLocal(context.VariableInjection);
        if (field.FieldType.IsValueType)
            context.Code.Emit(OpCodes.Unbox_Any, field.FieldType);
        context.Code.Emit(OpCodes.Stfld, field);
    }

    private static void EmitInjectProperty(in EmittingContext context, PropertyInfo property)
    {
        context.EmitLoadTarget();
        context.Code.LoadLocal(context.VariableInjection);
        if (property.PropertyType.IsValueType)
            context.Code.Emit(OpCodes.Unbox_Any, property.PropertyType);
        context.Code.Emit(property.SetMethod!.IsVirtual ? OpCodes.Callvirt : OpCodes.Call,
            property.SetMethod!);
    }
}