﻿using System.Text;
using Microsoft.Extensions.Logging;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing;
using Robotless.Modules.Utilities;
using Spectre.Console;
using Spectre.Console.Json;
using Spectre.Console.Rendering;
using Color = Spectre.Console.Color;
using Console = Spectre.Console.AnsiConsole;

namespace Robotless.Modules.Logging.Handlers;

public class ConsolePrinter(ISerializationProvider? serialization = null) : ILogHandler
{
    #region Style Settings
    
    public readonly Dictionary<LogLevel, Style> LevelToStyleMappings = new()
    {
        { LogLevel.None, Color.Grey },
        { LogLevel.Trace, Color.Grey },
        { LogLevel.Debug, Color.DarkGreen },
        { LogLevel.Information, Color.White },
        { LogLevel.Warning, Color.DarkOrange },
        { LogLevel.Error, Color.Red3_1 },
        { LogLevel.Critical, Color.Red1 }
    };

    public readonly Dictionary<LogLevel, string> LevelToTextMappings = new()
    {
        { LogLevel.None, "NONE".PadRight(5) },
        { LogLevel.Trace, "TRACE".PadRight(5) },
        { LogLevel.Debug, "DEBUG".PadRight(5) },
        { LogLevel.Information, "INFO".PadRight(5) },
        { LogLevel.Warning, "WARN".PadRight(5) },
        { LogLevel.Error, "ERROR".PadRight(5) },
        { LogLevel.Critical, "FATAL".PadRight(5) }
    };

    public Style StyleAuthorId { get; set; } = new(Color.MediumOrchid);

    public Style StyleAuthorName { get; set; } = new(Color.Orange3);

    public Style StyleScope { get; set; } = new();

    public Style StyleMessage { get; set; } = new();

    public Style StyleItem { get; set; } = new(Color.LightSteelBlue3);

    public ExceptionSettings ExceptionSettings { get; set; } = new()
    {
        Format = ExceptionFormats.ShortenMethods | ExceptionFormats.ShortenTypes
    };
    
    #endregion Style Settings

    /// <summary>
    /// Logs with level less than this will not be printed.
    /// </summary>
    public LogLevel MinimumLevel { get; set; } = LogLevel.Trace;
    
    private void AppendItemRows(Tree messageTree, string title, IRenderable content)
    {
        messageTree.Nodes.Add(new TreeNode(
            new Rows(
                new Text(title, StyleItem),
                content)
        ));
    }
        
    private void AppendItemColumns(Tree messageTree, string title, IRenderable content)
    {
        messageTree.Nodes.Add(new TreeNode(
            new Columns(
                    new Text(title, StyleItem),
                    content)
                { Expand = false }
        ));
    }
    
    private void FormatDetailsJson(Tree messageTree, ExpandoDictionary details)
    {
        var text = new StringWriter();
        JsonWriter? writer = null;
        foreach (var (name, value) in details)
        {
            switch (value)
            {
                case IRenderable renderable:
                    AppendItemRows(messageTree, $"{name}:", 
                        new Panel(renderable)
                        {
                            Border = BoxBorder.Rounded,
                        });
                    break;
                case string textItem:
                    AppendItemColumns(messageTree, $"{name}:", 
                        new Text(textItem));
                    break;
                case not null:
                    if (writer == null)
                    {
                        writer = new JsonWriter(text);
                        writer.WriteStartDocument();
                    }

                    writer.WriteName(name);
                    if (serialization == null)
                        writer.WriteString(value.ToString());
                    else
                        serialization.SaveSnapshot(value.GetType(), value, writer);
                    break;
            }
        }

        if (writer == null) 
            return;
        writer.WriteEndDocument();
        writer.Flush();
        AppendItemRows(messageTree, "Details:", new JsonText(text.ToString()));
    }
    
    public void HandleLog(LogItem log)
    {
        if (log.Level < MinimumLevel)
            return;
        
        var columns = new List<IRenderable>
        {
            new Text($"[{log.Time:HH:mm:ss.ffff}]"),
            new Text(LevelToTextMappings[log.Level], LevelToStyleMappings[log.Level]),
        };
        if (log.AuthorId != Guid.Empty)
            columns.Add(new Text($"#{log.AuthorId.ToString("N")[^8..]}", StyleAuthorId));
        if (!string.IsNullOrWhiteSpace(log.AuthorName))
            columns.Add(new Text($"\"{log.AuthorName}\"", StyleAuthorName));
        if (log.Scope != null)
        {
            var scopeText = new StringBuilder();
            var scopeStack = new Stack<LogScope>();
            for (var scope = log.Scope; scope != null; scope = scope.UpperScope)
                scopeStack.Push(scope);
            while (scopeStack.TryPop(out var scope))
                scopeText.Append($" [bold white]→[/] {scope.Message.EscapeMarkup()}");
            columns.Add(new Markup(scopeText.ToString(), StyleScope));
        }

        var messageTree = new Tree(new Columns(columns) { Expand = false });

        AppendItemColumns(messageTree, "Message:", new Text(log.Message, StyleMessage));

        if (log.Details.Count > 0)
            FormatDetailsJson(messageTree, log.Details);

        if (log.Exception != null)
            messageTree.AddNode(log.Exception.GetRenderable(ExceptionSettings));

        Console.Write(messageTree);

        Console.WriteLine();
    }
}