﻿using System.Collections.Concurrent;
using System.Threading.Channels;
using JetBrains.Annotations;
using Robotless.Kernel;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Logging;

public class LogCenter : WorkspaceService
{
    [PublicAPI] public IDictionary<Type, ILogHandler> LogHandlers { get; } 
        = new ConcurrentDictionary<Type, ILogHandler>();
    
    /// <summary>
    /// Submit a log item to the log center.
    /// By default, the log handling will be done in a background thread to avoid blocking the caller.
    /// </summary>
    /// <param name="log">Log item to submit and handle.</param>
    /// <param name="emergent">
    /// If true, the log handling will be done in the current thread immediately.
    /// It is useful if the caller is going to throw an exception.
    /// </param>
    public void SubmitLog(LogItem log, bool emergent = false)
    {
        if (emergent || !_channel.Writer.TryWrite(log))
            HandleLog(log);
    }

    private void HandleLog(LogItem log)
    {
        foreach (var (_, handler) in LogHandlers)
            handler.HandleLog(log);
        LogItem.Recycle(log);
    }
    
    private readonly Channel<LogItem> _channel = Channel.CreateUnbounded<LogItem>();

    private Task? _distributor;
    
    protected override Task Startup()
    {
        _distributor = Task.Factory.StartNew(async () =>
        {
            await foreach (var log in _channel.Reader.ReadAllAsync())
                HandleLog(log);
        }, TaskCreationOptions.LongRunning).Unwrap();
        
        return Task.CompletedTask;
    }

    protected override async Task Shutdown()
    {
        var distributor = Interlocked.Exchange(ref _distributor, null) ?? Task.CompletedTask;
        _channel.Writer.Complete();
        await distributor;
    }
}

public static class LogCenterExtensions
{
    public static LogCenter UseLogHandler(this LogCenter center, ILogHandler handler)
    {
        center.LogHandlers[handler.GetType()] = handler;
        return center;
    }
}