﻿using System.Collections.Concurrent;
using Microsoft.Extensions.Logging;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Logging;

public class LogItem
{
    private LogLevel _level;
    
    /// <summary>
    /// Importance level, which describes how important this log is.
    /// </summary>
    public LogLevel Level
    {
        get => _level;
        init => _level = value;
    }

    private string _message = string.Empty;

    /// <summary>
    /// Human-readable text message to describe what happened.
    /// </summary>
    public string Message
    {
        get => _message;
        init => _message = value;
    }

    private DateTime _time;
    
    /// <summary>
    /// Time when this log is created or submitted.
    /// </summary>
    public DateTime Time
    {
        get => _time;
        init => _time = value;
    }
    
    private Guid _authorId;
    
    /// <summary>
    /// GUID which represents the object which created this log or this log is mainly about.
    /// </summary>
    public Guid AuthorId
    {
        get => _authorId;
        init => _authorId = value;
    }
    
    private string? _authorName;

    /// <summary>
    /// Human-readable name of the author.
    /// This is only for display purposes, and should not be used to identify the author.
    /// </summary>
    public string? AuthorName
    {
        get => _authorName;
        init => _authorName = value;
    }
    
    private LogScope? _scope;

    public LogScope? Scope
    {
        get => _scope;
        init => _scope = value;
    }
    
    private Exception? _exception;

    /// <summary>
    /// The exception along with this log.
    /// </summary>
    public Exception? Exception
    {
        get => _exception;
        init => _exception = value;
    }
    
    /// <summary>
    /// Additional data that is not included in the log message,
    /// but can provide supplementary details about the log.
    /// </summary>
    public ExpandoDictionary Details { get; init; } = new();
    
    #region Pooling
    
    private static readonly List<KeyValuePair<string, object>> EmptyData = [];
    
    private static readonly ConcurrentBag<LogItem> LogPool = new();
    
    #endregion
    
    public static int MaxPooledCount { get; set; } = Environment.ProcessorCount * 2;

    // This flag is used to avoid a log item to be recycled multiple times.
    private bool _isRecycled = false;
    
    public static LogItem Allocate(LogLevel level, string message, 
        DateTime time, Guid authorId, string? authorName, LogScope? scope, Exception? exception)
    {
        if (!LogPool.TryTake(out var log))
            log = new LogItem();
        log._level = level;
        log._message = message;
        log._time = time;
        log._authorId = authorId;
        log._authorName = authorName;
        log._isRecycled = false;
        log._scope = scope;
        log._exception = exception;
        return log;
    }

    public static void Recycle(LogItem log)
    {
        if (log._isRecycled)
            return;
        log._isRecycled = true;
        log._level = LogLevel.Information;
        log._message = string.Empty;
        log._time = default;
        log._authorId = Guid.Empty;
        log._authorName = null;
        log._scope = null;
        log._exception = null;
        log.Details.Clear();
        
        if (LogPool.Count < MaxPooledCount)
            LogPool.Add(log);
    }
}