using JetBrains.Annotations;
using Microsoft.Extensions.Logging;
using Robotless.Framework;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Logging;

public class LoggerComponent : Component, ILogger
{
    public required LogCenter LogCenter { get; init; }
    
    /// <summary>
    /// GUID of the author to write in the log item.
    /// </summary>
    [PublicAPI] public Guid AuthorId => Entity.Identifier;

    /// <summary>
    /// Name of the author to write in the log item.
    /// </summary>
    [PublicAPI] public string? AuthorName { get; set; }
    
    /// <summary>
    /// The minimum level of logs to be submitted to the log handler.
    /// </summary>
    [PublicAPI]
    public LogLevel MinimumSubmissionLevel { get; set; } = LogLevel.Information;
    
    /// <summary>
    /// The minimum level of logs to be considered as emergent logs.
    /// </summary>
    [PublicAPI]
    public LogLevel MinimumEmergencyLevel { get; set; } = LogLevel.Critical;
    
    public bool IsEnabled(LogLevel logLevel) => logLevel >= MinimumSubmissionLevel;

    public IDisposable BeginScope<TState>(TState state) where TState : notnull
    {
        var scope = new LogScope(target =>
        {
            _scopes.Remove(target);
            var elapsedTime = target.EndingTime - target.BeginningTime;
            this.LogTrace("Scope End: \"{Message}\". Elapsed Time: {Duration}", 
                target.Message, elapsedTime.ToSmartString());
        })
        {
            BeginningTime = DateTime.Now,
            Message = state.ToString() ?? "Unknown",
            UpperScope = _scopes.Last?.Value
        };
        this.LogTrace("Scope Begin: \"{Message}\".", scope.Message);
        _scopes.AddLast(scope);
        return scope;
    }
    
    private readonly LinkedList<LogScope> _scopes = [];
    
    public void Log<TState>(LogLevel logLevel, EventId eventId, TState state, Exception? exception, 
        Func<TState, Exception?, string> formatter)
    {
        if (!IsEnabled(logLevel))
            return;
        var log = LogItem.Allocate(logLevel, formatter(state, null), 
            DateTime.Now, AuthorId, AuthorName,
            _scopes.Last?.Value, exception);
        LogCenter.SubmitLog(log, log.Level >= MinimumEmergencyLevel);
    }
    
    /// <summary>
    /// Log an exception.
    /// </summary>
    /// <param name="exception">Exception to log.</param>
    /// <param name="description">Description for the exception, such as when and where it occured.</param>
    /// <param name="level">
    /// Importance level of the log, which is <see cref="LogLevel.Error"/> by default.
    /// </param>
    /// <param name="details">Detailed data along with the exception.</param>
    public void LogException(Exception exception, string? description = null, 
        LogLevel level = LogLevel.Error, Action<dynamic>? details = null)
    {
        if (!IsEnabled(level)) 
            return;
        var log = LogItem.Allocate(level, description ?? "Exception Occurred.", 
            DateTime.Now, AuthorId, AuthorName, _scopes.Last?.Value, exception);
        details?.Invoke(log.Details);
        LogCenter.SubmitLog(log, log.Level >= MinimumEmergencyLevel);
    }
    
    public void LogDetails(LogLevel logLevel, string description, Action<dynamic> details)
    {
        if (!IsEnabled(logLevel)) 
            return;
        var log = LogItem.Allocate(logLevel, description, 
            DateTime.Now, AuthorId, AuthorName, _scopes.Last?.Value, null);
        details(log.Details);
        LogCenter.SubmitLog(log, log.Level >= MinimumEmergencyLevel);
    }
}

public static class LoggerComponentExtensions
{
    public static LoggerComponent? GetLogger(this Entity entity)
        => entity.GetComponent<LoggerComponent>();
    
    public static LoggerComponent RequireLogger(this Entity entity)
        => entity.RequireComponent<LoggerComponent>();

    public static LoggerComponent EnsureLogger(this Entity entity)
        => entity.EnsureComponent<LoggerComponent>();
    
    /// <summary>
    /// Log the exception and then return it.
    /// </summary>
    /// <param name="logger">Logger to use. If it is null, then the exception will not be logged.</param>
    /// <param name="exception">Exception to log.</param>
    /// <param name="description">Additional description.</param>
    /// <param name="level">Level of the log, which is <see cref="LogLevel.Error"/> by default.</param>
    /// <param name="details">Details about the log.</param>
    /// <typeparam name="TException">Exception type.</typeparam>
    /// <returns>The specified exception.</returns>
    public static TException Exception<TException>(this LoggerComponent? logger, TException exception,
        string? description = null, LogLevel level = LogLevel.Error, Action<dynamic>? details = null)
        where TException : Exception
    {
        logger?.LogException(exception, description, level, details);
        return exception;
    }
}