using System.Diagnostics;
using System.Runtime.CompilerServices;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing;

public interface ISerializationProvider
{
    /// <summary>
    /// Get a serializer instance for the specified target type.
    /// </summary>
    /// <param name="targetType">Type of the target value to serialize.</param>
    /// <returns>Serializer instance, or null if not found.</returns>
    ISnapshotSerializer? GetSerializer(Type targetType);
}

public static class SerializationProviderExtensions
{
    [DebuggerStepThrough, StackTraceHidden]
    public static ISnapshotSerializer<TTarget>? GetSerializer<TTarget>(this ISerializationProvider provider)
        => (ISnapshotSerializer<TTarget>?)provider.GetSerializer(typeof(TTarget));
    
    /// <summary>
    /// Require a serializer from this context.
    /// An exception will be thrown if the required serializer is not found.
    /// </summary>
    /// <param name="provider">Serialization provider to require serializer from.</param>
    /// <typeparam name="TTarget">Type of target value.</typeparam>
    /// <returns>Serializer for the target type.</returns>
    /// <exception cref="Exception">
    /// Thrown if a serializer for the target type cannot be found in this context.
    /// </exception>
    public static ISnapshotSerializer<TTarget> RequireSerializer<TTarget>(this ISerializationProvider provider)
        => (ISnapshotSerializer<TTarget>?)provider.GetSerializer(typeof(TTarget)) ??
           throw new Exception($"Cannot find required serializer for {typeof(TTarget)}.");

    /// <summary>
    /// Require a serializer from this context.
    /// An exception will be thrown if the required serializer is not found.
    /// </summary>
    /// <param name="provider">Serialization provider to require serializer from.</param>
    /// <param name="targetType">Type of target value.</param>
    /// <returns>Serializer for the target type.</returns>
    /// <exception cref="Exception">
    /// Thrown if a serializer for the target type cannot be found in this context.
    /// </exception>
    public static ISnapshotSerializer RequireSerializer(this ISerializationProvider provider, Type targetType)
        => provider.GetSerializer(targetType) ??
           throw new Exception($"Cannot find required serializer for \"{targetType}\".");

    /// <summary>
    /// Load a snapshot with a serializer found in this context.
    /// </summary>
    /// <param name="provider">Serialization provider to get serializer from.</param>
    /// <param name="target">Target value to load snapshot into.</param>
    /// <param name="reader">Reader to read snapshot from.</param>
    /// <typeparam name="TTarget">Type of the target value.</typeparam>
    public static void LoadSnapshot<TTarget>(this ISerializationProvider provider, ref TTarget target,
        SnapshotReader reader)
        => provider.RequireSerializer<TTarget>().LoadSnapshot(ref target, reader);

    /// <summary>
    /// Load a snapshot with a serializer of the specified type..
    /// </summary>
    /// <param name="provider">Serialization provider to get serializer from.</param>
    /// <param name="target">Target value to load snapshot into.</param>
    /// <param name="reader">Reader to read snapshot from.</param>
    /// <param name="targetType">Type of the target value.</param>
    public static void LoadSnapshot(this ISerializationProvider provider, Type targetType, ref object target,
        SnapshotReader reader)
    {
        var serializer = provider.RequireSerializer(targetType);
        serializer.LoadSnapshot(ref target, reader);
    }

    /// <summary>
    /// Save a snapshot with a serializer found in this context.
    /// </summary>
    /// <param name="provider">Serialization provider to get serializer from.</param>
    /// <param name="target">Target value to save snapshot of.</param>
    /// <param name="writer">Writer to write snapshot into.</param>
    /// <typeparam name="TTarget">Type of the target value.</typeparam>
    public static void SaveSnapshot<TTarget>(this ISerializationProvider provider, in TTarget target, IBsonWriter writer)
        => provider.RequireSerializer<TTarget>().SaveSnapshot(in target, writer);

    /// <summary>
    /// Save a snapshot with a serializer of the specified type..
    /// </summary>
    /// <param name="provider">Serialization provider to get serializer from.</param>
    /// <param name="target">Target value to save snapshot of.</param>
    /// <param name="writer">Writer to write snapshot into.</param>
    /// <param name="targetType">Type of the target value.</param>
    public static void SaveSnapshot(this ISerializationProvider provider, Type targetType, object target,
        IBsonWriter writer)
    {
        var serializer = provider.RequireSerializer(targetType);
        serializer.SaveSnapshot(target, writer);
    }

    /// <summary>
    /// Generate a JSON schema for the target type into the specified schema writer.
    /// </summary>
    /// <param name="provider">Serialization provider to get serializer from.</param>
    /// <param name="writer">Writer to write schema into.</param>
    /// <typeparam name="TTarget">Target value type to generate schema for.</typeparam>
    [DebuggerStepThrough, StackTraceHidden]
    public static void GenerateJsonSchema<TTarget>(this ISerializationProvider provider, SchemaWriter writer)
        => provider.RequireSerializer<TTarget>().GenerateJsonSchema(writer);

    /// <summary>
    /// Write a member to the writer. The name of the field is the same to the member name.
    /// </summary>
    /// <param name="provider">Serialization provider to use.</param>
    /// <param name="member">Member to save snapshot of.</param>
    /// <param name="writer">Writer to write snapshot into.</param>
    /// <param name="name">Name. Compiler will automatically set it to the name of the member.</param>
    /// <typeparam name="TTarget">Type of the value to save snapshot.</typeparam>
    public static void SaveMember<TTarget>(this ISerializationProvider provider,
        in TTarget member, IBsonWriter writer, [CallerMemberName] string name = "")
    {
        writer.WriteName(name);
        provider.SaveSnapshot(member, writer);
    }

    /// <summary>
    /// Load a member from the reader. The name of the field is the same to the member name.
    /// </summary>
    /// <param name="provider">Serialization provider to use.</param>
    /// <param name="member">Member to load snapshot into.</param>
    /// <param name="reader">Reader to reader snapshot from.</param>
    /// <param name="name">Name. Compiler will automatically set it to the name of the member.</param>
    /// <typeparam name="TTarget">Type of the value to load snapshot.</typeparam>
    public static void LoadMember<TTarget>(this ISerializationProvider provider,
        ref TTarget member, SnapshotReader reader, [CallerMemberName] string name = "")
    {
        reader.Locate(name);
        provider.LoadSnapshot(ref member, reader);
    }
}