using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing;

public interface ISnapshotSerializer
{
    /// <summary>
    /// Instantiate a new instance of the target type.
    /// </summary>
    /// <param name="target">Target value to be overwritten with the new instance.</param>
    void NewInstance(out object target);

    /// <summary>
    /// Write the schema of the snapshot into the specified writer.
    /// </summary>
    /// <param name="schema">Writer to write schema into.</param>
    void GenerateJsonSchema(SchemaWriter schema);
    
    /// <summary>
    /// Load a snapshot to the specified target instance from the specified reader.
    /// </summary>
    /// <param name="target">Target to load snapshot into.</param>
    /// <param name="reader">Snapshot reader to read snapshot from.</param>
    void LoadSnapshot(ref object target, SnapshotReader reader);

    /// <summary>
    /// Save a snapshot of the specified target instance into the specified writer.
    /// </summary>
    /// <param name="target">Target to save snapshot of.</param>
    /// <param name="writer">Snapshot write to write snapshot into.</param>
    void SaveSnapshot(in object target, IBsonWriter writer);
    
}

public interface ISnapshotSerializer<TTarget> : ISnapshotSerializer
{
    void ISnapshotSerializer.NewInstance(out object target)
    {
        NewInstance(out var castedTarget);
        target = castedTarget!;
    }

    void ISnapshotSerializer.LoadSnapshot(ref object target, SnapshotReader reader)
    {
        if (target is TTarget castedTarget)
        {
            LoadSnapshot(ref castedTarget, reader);
            return;
        }
        NewInstance(out var newTarget);
        LoadSnapshot(ref newTarget, reader);
        target = newTarget!;
    }

    void ISnapshotSerializer.SaveSnapshot(in object target, IBsonWriter writer)
    {
        SaveSnapshot((TTarget)target, writer);
    }

    /// <inheritdoc cref="ISnapshotSerializer.NewInstance"/>
    void NewInstance(out TTarget target);
    
    /// <inheritdoc cref="ISnapshotSerializer.LoadSnapshot"/>
    void LoadSnapshot(ref TTarget target, SnapshotReader reader);

    /// <inheritdoc cref="ISnapshotSerializer.SaveSnapshot"/>
    void SaveSnapshot(in TTarget target, IBsonWriter writer);
}