using System.Collections.Concurrent;
using System.Reflection;
using Robotless.Modules.Injecting;
using Robotless.Modules.Serializing.Serializers.BsonValues;
using Robotless.Modules.Serializing.Serializers.Containers;
using Robotless.Modules.Serializing.Serializers.Embedded;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing;

public class SerializationContext : ISerializationContext, IInjectionProvider
{
    private readonly ConcurrentDictionary<Type, Type> _specificSerializerTypes = new();

    private readonly ConcurrentDictionary<Type, Type> _genericSerializerTypes = new();

    private readonly ConcurrentDictionary<Type, ISnapshotSerializer> _cachedSerializers = new();

    public ICollection<ISerializerTypeProvider> TypeProviders { get; } =
        new LinkedList<ISerializerTypeProvider>();

    /// <summary>
    /// Set a serializer type for the specified target type.
    /// Context will automatically instantiate the serializer type when required.
    /// </summary>
    /// <param name="targetType">Type of the target value to serialize.</param>
    /// <param name="serializerType">Type of serializer to use.</param>
    public virtual void SetSerializer(Type targetType, Type? serializerType)
    {
        if (serializerType == null)
        {
            _specificSerializerTypes.TryRemove(targetType, out _);
            return;
        }

        var serializerInterfaceType = MatchSerializerInterface(serializerType);
        if (serializerInterfaceType == null)
            throw new ArgumentException(
                $"Target type \"{serializerType.Name}\" does not implement interface ISnapshotSerializer<>.");
        if (serializerType.IsInterface || serializerType.IsAbstract)
            throw new ArgumentException($"Serializer type \"{serializerType.Name}\" is an interface or abstract class.");
        var serializerTargetType = serializerInterfaceType.GetGenericArguments()[0];
        
        if (!targetType.IsGenericTypeDefinition)
        {
            if (targetType != serializerTargetType)
                throw new ArgumentException(
                    $"Serializer type \"{serializerType.Name}\" cannot be registered " +
                    $"for target type \"{targetType.Name}\".");
            _specificSerializerTypes[targetType] = serializerType;
            return;
        }

        // Validate generic serializer type.
        if (!serializerType.IsGenericTypeDefinition)
            throw new ArgumentException($"Non-generic-definition serializer type \"{serializerType.Name}\" " +
                                        $"cannot be registered for target type \"{targetType.Name}\".");
        if (targetType != serializerTargetType.GetGenericTypeDefinition())
            throw new ArgumentException(
                $"Generic target type of serializer \"{serializerType.Name}\" does not match " +
                $"target type {targetType.Name}.");

        _genericSerializerTypes[targetType] = serializerType;
    }

    public virtual ISnapshotSerializer? GetSerializer(Type targetType)
    {
        if (_cachedSerializers.TryGetValue(targetType, out var serializer))
            return serializer;
        var serializerType = SearchSerializerType(targetType);
        if (serializerType == null)
            return null;
        serializer = (ISnapshotSerializer)this.NewObject(serializerType);
        _cachedSerializers[targetType] = serializer;
        return serializer;
    }

    /// <summary>
    /// This method will discard all cached serializer instances,
    /// and force the context to re-instantiate serializers when required.
    /// It is helpful when fundamental serializers are changed in this context. <br/>
    /// For example, instance serializerA (of type SerializerA) depends on the serializer for integer32,
    /// and when it is initialized, the serializer type for integer32 is set to SerializerB;
    /// then the serializer type for integer32 is set to SerializerC,
    /// however, serializerA's dependency will not automatically update to SerializerC. <br/>
    /// Users should invoke this method and re-instantiate SerializerA to get an instance of it
    /// with the dependency to SerializerC. <br/>
    /// Otherwise, the context will continue to provide the cached instance of SerializerA
    /// with the dependency to SerializerB.
    /// </summary>
    public virtual void ClearCachedSerializerInstances()
    {
        _cachedSerializers.Clear();
    }

    protected virtual Type? SearchSerializerType(Type targetType)
    {
        // Handle serializer designation attribute.
        if (targetType.GetCustomAttribute<UsingSnapshotSerializerAttribute>() is { } attribute)
            return attribute.SerializerType;

        // Handle array types.
        if (targetType.IsArray)
        {
            var arrayRank = targetType.GetArrayRank();
            if (arrayRank == 1)
                return typeof(ArraySerializer<>).MakeGenericType(targetType.GetElementType()!);
            return typeof(MatrixSerializer<,>)
                .MakeGenericType(targetType, targetType.GetElementType()!);
        }

        // Search non-generic serializers.
        if (_specificSerializerTypes.TryGetValue(targetType, out var serializerType))
            return serializerType;

        // Search generic serializers.
        if (targetType.IsGenericType &&
            _genericSerializerTypes.TryGetValue(targetType.GetGenericTypeDefinition(), out serializerType))
        {
            var serializerTargetType = serializerType.GetInterface(typeof(ISnapshotSerializer<>).Name)?
                .GetGenericArguments()[0]!;
            var genericParameters = new Type[targetType.GetGenericArguments().Length];
            GenericParameterInjector.Inject(targetType, serializerTargetType, genericParameters);
            return serializerType.MakeGenericType(genericParameters);
        }

        foreach (var typeProvider in TypeProviders)
        {
            serializerType = typeProvider.GetSerializerType(targetType);
            if (serializerType != null)
                return serializerType;
        }

        return null;
    }

    private static Type? MatchSerializerInterface(Type serializerType)
        => serializerType.GetInterfaces().FirstOrDefault(
            interfaceType => interfaceType.IsGenericType &&
                             interfaceType.GetGenericTypeDefinition() == typeof(ISnapshotSerializer<>));

    /// <summary>
    /// Serialization context can only provide itself and ISnapshotSerializer{TTarget} instances as injections.
    /// </summary>
    /// <param name="type">Must be types of ISnapshotSerializer{TTarget}.</param>
    /// <param name="key">This argument is ignored.</param>
    /// <param name="requester"></param>
    /// <returns>Snapshot serializer, or null if not found.</returns>
    object? IInjectionProvider.GetInjection(Type type, object? key, InjectionRequester requester)
    {
        if (type.IsAssignableFrom(typeof(SerializationContext)))
            return this;
        if (!type.IsGenericType || type.GetGenericTypeDefinition() != typeof(ISnapshotSerializer<>))
            throw new ArgumentException(
                $"Invalid injection type: \"{type.Name}\": " +
                "SerializerContext cannot provide injections other than itself and ISnapshotSerializer<TTarget>.");
        var targetType = type.GetGenericArguments()[0];
        return GetSerializer(targetType);
    }

    /// <summary>
    /// Currently, there is no need for serialization context to support scopes.
    /// </summary>
    /// <returns>This serialization context.</returns>
    IInjectionProvider IInjectionProvider.NewScope() => this;
}

public static class SerializationContextFactory
{
    public static TInjectionContainer AddSerialization<TInjectionContainer>(
        this TInjectionContainer container, Action<SerializationContext, IInjectionProvider> configure)
        where TInjectionContainer : IInjectionContainer
    {
        container.AddSingleton((provider, _) =>
        {
            var serialization = new SerializationContext();
            configure(serialization, provider);
            return serialization;
        });
        container.AddRedirection<ISerializationContext, SerializationContext>();
        container.AddRedirection<ISerializationProvider, SerializationContext>();
        return container;
    }
}