﻿using System.Reflection;
using System.Reflection.Emit;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;
using Robotless.Modules.Utilities;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Serializing;

public partial class SerializerGenerator
{
    private class LoaderMethodBuilder : ISerializerMethodBuilder
    {
        private readonly ClassContext _context;
        
        private readonly ILGenerator _code;

        private readonly MethodBuilder _method;
        
        public LoaderMethodBuilder(ClassContext context)
        {
            _context = context;
            _method = _context.TypeBuilder.DefineMethod("OnLoadSnapshot",
                MethodAttributes.Family | MethodAttributes.Virtual | MethodAttributes.HideBySig,
                CallingConventions.Standard,
                typeof(void), null, null,
                [_context.TargetType.MakeByRefType(), typeof(SnapshotReader)],
                [Type.EmptyTypes, Type.EmptyTypes],
                [Type.EmptyTypes, Type.EmptyTypes]);
            _method.DefineParameter(1, ParameterAttributes.None, "target");
            _method.DefineParameter(2, ParameterAttributes.None, "reader");
            _code = _method.GetILGenerator();

            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Callvirt,
                typeof(SnapshotReader).GetMethod(nameof(SnapshotReader.ReadStartDocument))!);
        }

        public void Build()
        {
            if (_context.TargetType.IsAssignableTo(typeof(ICustomizedSerialization)))
            {
                EmitLoadTarget();
                _code.Emit(OpCodes.Ldarg_2);
                _code.Emit(OpCodes.Ldarg_0);
                _code.Emit(OpCodes.Callvirt,
                    typeof(SnapshotSerializerBase<>)
                        .MakeGenericType(_context.TargetType)
                        .GetProperty(nameof(SnapshotSerializerBase<object>.Provider))!.GetMethod!);
                _code.Emit(OpCodes.Callvirt,
                    typeof(ICustomizedSerialization).GetMethod(nameof(ICustomizedSerialization.OnLoadSnapshot))!);
            }
            
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Callvirt,
                typeof(SnapshotReader).GetMethod(nameof(SnapshotReader.ReadEndDocument))!);

            _code.Emit(OpCodes.Ret);
            
            _context.TypeBuilder.DefineMethodOverride(_method,
                _context.BaseType.GetMethod($"On{nameof(SnapshotSerializerBase<object>.LoadSnapshot)}", 
                    BindingFlags.NonPublic | BindingFlags.Instance)!);
        }

        public void GenerateForField(FieldInfo field)
        {
            // Locate the data entry for this member.
            // Currently, generated snapshot loader can tolerate missing data entries for members.
            _code.If(predicate: _ => { EmitTryLocateEntry(field.Name); },
                whenTrue: _ =>
                {
                    _context.EmitLoadSerializer(_code, field.FieldType);

                    EmitLoadTarget();
                    _code.Emit(OpCodes.Ldflda, field);

                    _code.Emit(OpCodes.Ldarg_2);
                    _code.Emit(OpCodes.Callvirt,
                        typeof(ISnapshotSerializer<>).MakeGenericType(field.FieldType)
                            .GetMethod(nameof(ISnapshotSerializer<object>.LoadSnapshot))!);
                }, whenFalse: _ => {});
        }

        public void GenerateForProperty(PropertyInfo property)
        {
            // Locate the data entry for this member.
            // Currently, generated snapshot loader can tolerate missing data entries for members.
            _code.If(predicate: _ => { EmitTryLocateEntry(property.Name); },
                whenTrue: _ =>
                {
                    // Store the property value into a local variable.
                    var variable = _code.DeclareLocal(property.PropertyType);
                    EmitLoadTarget();
                    _code.Emit(OpCodes.Callvirt, property.GetMethod!);
                    _code.Emit(OpCodes.Stloc, variable);

                    // Invoke the snapshot loader method.
                    _context.EmitLoadSerializer(_code, property.PropertyType);
                    _code.Emit(OpCodes.Ldloca, variable);
                    _code.Emit(OpCodes.Ldarg_2);
                    _code.Emit(OpCodes.Callvirt,
                        typeof(ISnapshotSerializer<>).MakeGenericType(property.PropertyType)
                            .GetMethod(nameof(ISnapshotSerializer<object>.LoadSnapshot))!);

                    // Store the local variable back to the property.
                    EmitLoadTarget();
                    _code.Emit(OpCodes.Ldloc, variable);
                    _code.Emit(OpCodes.Callvirt, property.SetMethod!);
                }, whenFalse: _ => { });
        }
        
        private void EmitLoadTarget()
        {
            _code.Emit(OpCodes.Ldarg_1);
            if (!_context.TargetType.IsValueType)
                _code.Emit(OpCodes.Ldind_Ref);
        }

        private void EmitTryLocateEntry(string name)
        {
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Ldstr, name);
            _code.Emit(OpCodes.Callvirt,
                typeof(SnapshotReader).GetMethod(nameof(SnapshotReader.TryLocate))!);
        }
    }
}