﻿using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using MongoDB.Bson.IO;

namespace Robotless.Modules.Serializing;

public partial class SerializerGenerator
{
    private class SaverMethodBuilder : ISerializerMethodBuilder
    {
        private readonly ClassContext _context;
        
        private readonly ILGenerator _code;

        private readonly MethodBuilder _method;
        
        public SaverMethodBuilder(ClassContext context)
        {
            _context = context;
            _method = _context.TypeBuilder.DefineMethod("OnSaveSnapshot",
                MethodAttributes.Family | MethodAttributes.Virtual | MethodAttributes.HideBySig,
                CallingConventions.Standard,
                typeof(void), null, null,
                [_context.TargetType.MakeByRefType(), typeof(IBsonWriter)],
                [[typeof(InAttribute)], Type.EmptyTypes], null);
            var parameterTarget = _method.DefineParameter(1, ParameterAttributes.In, "target");
            parameterTarget.SetCustomAttribute(AttributeIn);
            parameterTarget.SetCustomAttribute(AttributeReadOnly);
            _method.DefineParameter(2, ParameterAttributes.None, "writer");
            _code = _method.GetILGenerator();

            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Callvirt,
                typeof(IBsonWriter).GetMethod(nameof(IBsonWriter.WriteStartDocument))!);
        }
        
        public void Build()
        {
            if (_context.TargetType.IsAssignableTo(typeof(ICustomizedSerialization)))
            {
                EmitLoadTarget();
                _code.Emit(OpCodes.Ldarg_2);
                _code.Emit(OpCodes.Ldarg_0);
                _code.Emit(OpCodes.Callvirt,
                    typeof(SnapshotSerializerBase<>)
                        .MakeGenericType(_context.TargetType)
                        .GetProperty(nameof(SnapshotSerializerBase<object>.Provider))!.GetMethod!);
                _code.Emit(OpCodes.Callvirt, 
                    typeof(ICustomizedSerialization).
                        GetMethod(nameof(ICustomizedSerialization.OnSaveSnapshot))!);
            }
        
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Callvirt,
                typeof(IBsonWriter).GetMethod(nameof(IBsonWriter.WriteEndDocument))!);
            _code.Emit(OpCodes.Ret);
            
            _context.TypeBuilder.DefineMethodOverride(_method,
                _context.BaseType.GetMethod($"On{nameof(SnapshotSerializerBase<object>.SaveSnapshot)}",
                    BindingFlags.NonPublic | BindingFlags.Instance)!);
        }

        public void GenerateForField(FieldInfo field)
        {
            // Write name.
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Ldstr, field.Name);
            _code.Emit(OpCodes.Callvirt,
                typeof(IBsonWriter).GetMethod(nameof(IBsonWriter.WriteName))!);
        
            // Write value.
            _context.EmitLoadSerializer(_code, field.FieldType);
            EmitLoadTarget();
            _code.Emit(OpCodes.Ldflda, field);
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Callvirt,
                typeof(ISnapshotSerializer<>).MakeGenericType(field.FieldType)
                    .GetMethod(nameof(ISnapshotSerializer<object>.SaveSnapshot))!);
        }

        public void GenerateForProperty(PropertyInfo property)
        {
            // Write property name.
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Ldstr, property.Name);
            _code.Emit(OpCodes.Callvirt,
                typeof(IBsonWriter).GetMethod(nameof(IBsonWriter.WriteName))!);

            // Load property value into a local variable.
            var variable = _code.DeclareLocal(property.PropertyType);
            EmitLoadTarget();
            _code.Emit(OpCodes.Callvirt, property.GetMethod!);
            _code.Emit(OpCodes.Stloc, variable);

            // Write variable value into writer.
            _context.EmitLoadSerializer(_code, property.PropertyType);
            _code.Emit(OpCodes.Ldloca, variable);
            _code.Emit(OpCodes.Ldarg_2);
            _code.Emit(OpCodes.Callvirt,
                typeof(ISnapshotSerializer<>).MakeGenericType(property.PropertyType)
                    .GetMethod(nameof(ISnapshotSerializer<object>.SaveSnapshot))!);
        }
        
        private void EmitLoadTarget()
        {
            _code.Emit(OpCodes.Ldarg_1);
            if (!_context.TargetType.IsValueType)
                _code.Emit(OpCodes.Ldind_Ref);
        }
    }
    
    private static readonly CustomAttributeBuilder AttributeIn = new(
        typeof(InAttribute).GetConstructor(Type.EmptyTypes)!, []);

    private static readonly CustomAttributeBuilder AttributeReadOnly = new(
        typeof(IsReadOnlyAttribute).GetConstructor(Type.EmptyTypes)!, []);
}