﻿using System.Reflection;
using System.Reflection.Emit;
using Robotless.Modules.Documenting;
using Robotless.Modules.Serializing.Utilities;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Serializing;

public partial class SerializerGenerator
{
    private class SchemaMethodBuilder : ISerializerMethodBuilder
    {
        private readonly ClassContext _context;
        
        private readonly ILGenerator _code;

        private readonly MethodBuilder _method;

        private readonly IDocumentation? _documentation;

        public SchemaMethodBuilder(ClassContext context, IDocumentation? documentation)
        {
            _context = context;
            _documentation = documentation;
            
            _method = _context.TypeBuilder.DefineMethod("GenerateJsonSchema",
                MethodAttributes.Public | MethodAttributes.Virtual | MethodAttributes.HideBySig,
                CallingConventions.Standard,
                typeof(void), [typeof(SchemaWriter)]);

            _code = _method.GetILGenerator();
            
            _code.Emit(OpCodes.Ldarg_1);
            _code.Emit(OpCodes.Ldc_I4, (int) SchemaType.Object);
            _code.Emit(OpCodes.Callvirt, 
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.DefineType))!);
        
            EmitWriteName("properties");
        
            EmitWriteStartDocument();
        }
        
        public void Build()
        {
            EmitWriteEndDocument(); // End of properties.
        
            // Currently all field or property are considered as 'required'.
            _code.LoadArgument1();
            _code.Emit(OpCodes.Ldc_I4, RequiredProperties.Count);
            _code.Emit(OpCodes.Newarr, typeof(string));
            for (var index = 0; index < RequiredProperties.Count; ++index)
            {
                _code.Emit(OpCodes.Dup);
                _code.Emit(OpCodes.Ldc_I4, index);
                _code.Emit(OpCodes.Ldstr, RequiredProperties[index]);
                _code.Emit(OpCodes.Stelem_Ref);
            }
            _code.CallVirtual(typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.DefineRequiredProperties))!);
        
            // Currently, no additional properties are allowed.
            _code.LoadArgument1();
            _code.CallVirtual(
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.DefineNoAdditionalPropertiesAllowed))!);
        
            _code.MethodReturn();
            
            _context.TypeBuilder.DefineMethodOverride(_method,
                _context.BaseType.GetMethod($"{nameof(SnapshotSerializerBase<object>.GenerateJsonSchema)}")!);

        }

        public void GenerateForField(FieldInfo field)
        {
            RequiredProperties.Add(field.Name);
            EmitWriteName(field.Name);
            EmitWriteStartDocument();
            var description = _documentation?.GetEntry(field);
            if (description != null)
            {
                var text = string.Empty;
                if (description.Summary != null)
                    text += description.Summary + "\n";
                if (description.Remarks != null)
                    text += description.Remarks + "\n";
                if (description.Example != null)
                    text += description.Example + "\n";
                EmitWriteDescription(text);
            }
                
            _context.EmitLoadSerializer(_code, field.FieldType);
            _code.LoadArgument1();
            _code.CallVirtual(typeof(ISnapshotSerializer)
                .GetMethod(nameof(ISnapshotSerializer.GenerateJsonSchema))!);
            EmitWriteEndDocument();
        }

        public void GenerateForProperty(PropertyInfo property)
        {
            RequiredProperties.Add(property.Name);
            EmitWriteName(property.Name);
            EmitWriteStartDocument();
            
            var description = _documentation?.GetEntry(property);
            if (description != null)
            {
                var text = string.Empty;
                if (description.Summary != null)
                    text += description.Summary + "\n";
                if (description.Remarks != null)
                    text += description.Remarks + "\n";
                if (description.Example != null)
                    text += description.Example + "\n";
                EmitWriteDescription(text);
            }
            
            _context.EmitLoadSerializer(_code, property.PropertyType);
            _code.LoadArgument1();
            _code.CallVirtual(typeof(ISnapshotSerializer)
                .GetMethod(nameof(ISnapshotSerializer.GenerateJsonSchema))!);
            EmitWriteEndDocument();
        }
        
        public readonly List<string> RequiredProperties = new();

        public void EmitWriteStartDocument()
        {
            _code.Emit(OpCodes.Ldarg_1);
            _code.Emit(OpCodes.Callvirt, 
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.WriteStartDocument))!);
        }
        
        public void EmitWriteEndDocument()
        {
            _code.Emit(OpCodes.Ldarg_1);
            _code.Emit(OpCodes.Callvirt, 
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.WriteEndDocument))!);
        }
        
        public void EmitWriteName(string name)
        {
            _code.Emit(OpCodes.Ldarg_1);
            _code.Emit(OpCodes.Ldstr, name);
            _code.Emit(OpCodes.Callvirt, 
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.WriteName))!);
        }
        
        public void EmitWriteDescription(string description)
        {
            _code.Emit(OpCodes.Ldarg_1);
            _code.Emit(OpCodes.Ldstr, description);
            _code.Emit(OpCodes.Callvirt, 
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.DefineDescription))!);
        }
        
        public void EmitWriteString(string text)
        {
            _code.Emit(OpCodes.Ldarg_1);
            _code.Emit(OpCodes.Ldstr, text);
            _code.Emit(OpCodes.Callvirt, 
                typeof(SchemaWriter).GetMethod(nameof(SchemaWriter.WriteString))!);
        }
    }
}